#include "ArithmeticDecoder.h"

#include <stdio.h>
#include <stdbool.h>

struct InputStream
{
	uint8_t *buffer;
	int pos;
};

struct KnownState
{
	int n,s;
	uint8_t yn,mps;
	int16_t lr,lrm;
	uint32_t x,dx;
	uint8_t k;
	int16_t lp,lx;
} *knownstates[];

uint8_t *streams[];

int numsteps[];

static size_t ReadFunc(void *context,uint8_t *buffer,size_t count)
{
	struct InputStream *stream=context;
	*buffer=stream->buffer[stream->pos++];
	return 1;
}

int main()
{
	int numtests=8;
	int errors=0;

	for(int i=0;i<numtests;i++)
	{
		printf("Running test %d...\n",i);

		struct InputStream stream=
		{
			.buffer=streams[i],
			.pos=0
		};

		WinZipJPEGArithmeticDecoder decoder;
		InitializeWinZipJPEGArithmeticDecoder(&decoder,ReadFunc,&stream);

		WinZipJPEGContext contexts[256];
		for(int j=0;j<sizeof(contexts)/sizeof(contexts[0]);j++) InitializeWinZipJPEGContext(&contexts[j]);

		int errors=0;
		if(decoder.lr!=knownstates[i][0].lr) { fprintf(stderr,"Wrong lr value (%x!=%x) at step %d!\n",decoder.lr,knownstates[i][0].lr,0); errors++; }
		if(decoder.lrm!=knownstates[i][0].lrm) { fprintf(stderr,"Wrong lrm value (%x!=%x) at step %d!\n",decoder.lrm,knownstates[i][0].lrm,0); errors++; }
		if(decoder.x!=knownstates[i][0].x) { fprintf(stderr,"Wrong x value (%x!=%x) at step %d!\n",decoder.x,knownstates[i][0].x,0); errors++; }

		for(int j=1;j<=numsteps[i];j++)
		{
			WinZipJPEGContext *context=&contexts[knownstates[i][j].s];
			int bit=NextBitFromWinZipJPEGArithmeticDecoder(&decoder,context);

			if(bit!=knownstates[i][j].yn) { fprintf(stderr,"Wrong yn value (%x!=%x) at step %d!\n",bit,knownstates[i][j].yn,j); errors++; }
			if(context->mps!=knownstates[i][j].mps) { fprintf(stderr,"Wrong mps value (%x!=%x) at step %d!\n",context->mps,knownstates[i][j].mps,j); errors++; }
			if(decoder.lr!=knownstates[i][j].lr) { fprintf(stderr,"Wrong lr value (%x!=%x) at step %d!\n",decoder.lr,knownstates[i][j].lr,j); errors++; }
			if(decoder.lrm!=knownstates[i][j].lrm) { fprintf(stderr,"Wrong lrm value (%x!=%x) at step %d!\n",decoder.lrm,knownstates[i][j].lrm,j); errors++; }
			if(decoder.x!=knownstates[i][j].x) { fprintf(stderr,"Wrong x value (%x!=%x) at step %d!\n",decoder.x,knownstates[i][j].x,j); errors++; }
			if(decoder.dx!=knownstates[i][j].dx) { fprintf(stderr,"Wrong dx value (%x!=%x) at step %d!\n",decoder.dx,knownstates[i][j].dx,j); errors++; }
			if(context->k!=knownstates[i][j].k) { fprintf(stderr,"Wrong k value (%x!=%x) at step %d!\n",context->k,knownstates[i][j].k,j); errors++; }
			if(decoder.lp!=knownstates[i][j].lp) { fprintf(stderr,"Wrong lp value (%x!=%x) at step %d!\n",decoder.lp,knownstates[i][j].lp,j); errors++; }
			if(decoder.lx!=knownstates[i][j].lx) { fprintf(stderr,"Wrong lx value (%x!=%x) at step %d!\n",decoder.lx,knownstates[i][j].lx,j); errors++; }
			if(errors>20) return 1;
		}
	}

	if(errors) return 1;

	printf("Decoder passed all tests.\n");
	return 0;
}


uint8_t stream0[]={ 0xef,0xd9,0x81,0xa3,0x73,0x2d,0x10,0x63,0xd4,0x69,0x2b,0xf6,0x0a,0xea,0xdf,0x0a,0x4e,0x9f,0xc5,0x80 };

struct KnownState knownstates0[257]=
{
	{ 0,-1,-1,-1,0x1001,0x1001,0x0000efd9,-1,-1,-1,-1 }, // This line seems to be wrong in the original patent.
	{ 1,0,1,0,0x1401,0x5001,0x00006fe5,0x007ff4,0x01,0x400,0x14c8 },
	{ 2,0,1,0,0x1801,0x5001,0x00002feb,0x003ffa,0x02,0x400,0x19ad },
	{ 3,0,1,0,0x1c01,0x5001,0x00000fee,0x001ffd,0x03,0x400,0x2000 },
	{ 4,0,0,0,0x0001,0x3001,0x000fee81,0x000000,0x03,0x400,0x0007 },
	{ 5,0,1,0,0x0401,0x3001,0x0007ef41,0x07ff40,0x04,0x400,0x040d },
	{ 6,0,1,0,0x0801,0x3001,0x0003efa1,0x03ffa0,0x05,0x400,0x0819 },
	{ 7,0,1,0,0x0c01,0x3001,0x0001efd1,0x01ffd0,0x06,0x400,0x0c30 },
	{ 8,0,1,0,0x1001,0x3001,0x0000efe9,0x00ffe8,0x07,0x400,0x1061 },
	{ 9,0,1,0,0x1401,0x3001,0x00006ff5,0x007ff4,0x08,0x400,0x14c7 },
	{ 10,0,1,0,0x1801,0x3001,0x00002ffb,0x003ffa,0x09,0x400,0x19ab },
	{ 11,0,1,0,0x1c01,0x3001,0x00000ffe,0x001ffd,0x0a,0x400,0x2000 },
	{ 12,0,1,1,0x0001,0x3b02,0x00000023,0x0ffe80,0x00,0x31b,0x2000 },
	{ 13,0,1,1,0x031c,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 14,0,1,1,0x0637,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 15,0,1,1,0x0952,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 16,0,1,1,0x0c6d,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 17,0,1,1,0x0f88,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 18,0,1,1,0x12a3,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 19,0,1,1,0x15be,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 20,0,1,1,0x18d9,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 21,0,1,1,0x1bf4,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 22,0,1,1,0x1f0f,0x3b02,0x00000023,0x000000,0x00,0x31b,0x2000 },
	{ 23,0,1,1,0x022a,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 24,0,1,1,0x0545,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 25,0,1,1,0x0860,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 26,0,1,1,0x0b7b,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 27,0,1,1,0x0e96,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 28,0,1,1,0x11b1,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 29,0,1,1,0x14cc,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 30,0,1,1,0x17e7,0x1b02,0x00002373,0x000000,0x00,0x31b,0x1b6a },
	{ 31,0,1,1,0x1b02,0x4b21,0x00002373,0x000000,0x00,0x07f,0x1b6a },
	{ 32,0,0,1,0x296f,0x590f,0x0000008d,0x0022e6,0x01,0x07f,0x2000 },
	{ 33,0,1,1,0x09ee,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 34,0,1,1,0x0a6d,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 35,0,1,1,0x0aec,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 36,0,1,1,0x0b6b,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 37,0,1,1,0x0bea,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 38,0,1,1,0x0c69,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 39,0,1,1,0x0ce8,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 40,0,1,1,0x0d67,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 41,0,1,1,0x0de6,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 42,0,1,1,0x0e65,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 43,0,1,1,0x0ee4,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 44,0,1,1,0x0f63,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 45,0,1,1,0x0fe2,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 46,0,1,1,0x1061,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 47,0,1,1,0x10e0,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 48,0,1,1,0x115f,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 49,0,1,1,0x11de,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 50,0,1,1,0x125d,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 51,0,1,1,0x12dc,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 52,0,1,1,0x135b,0x390f,0x00008d2d,0x000000,0x01,0x07f,0x1371 },
	{ 53,0,0,1,0x21c8,0x46fd,0x000009c5,0x008368,0x02,0x07f,0x2000 },
	{ 54,0,1,1,0x0247,0x26fd,0x0009c510,0x000000,0x02,0x07f,0x02da },
	{ 55,0,1,1,0x02c6,0x26fd,0x0009c510,0x000000,0x02,0x07f,0x02da },
	{ 56,0,0,1,0x1133,0x34eb,0x0000af10,0x091600,0x03,0x07f,0x1233 },
	{ 57,0,1,1,0x11b2,0x34eb,0x0000af10,0x000000,0x03,0x07f,0x1233 },
	{ 58,0,1,1,0x1231,0x34eb,0x0000af10,0x000000,0x03,0x07f,0x1233 },
	{ 59,0,0,1,0x209e,0x42d9,0x00000e48,0x00a0c8,0x04,0x07f,0x2000 },
	{ 60,0,0,1,0x0f0b,0x30c7,0x000115e3,0x0d3280,0x05,0x07f,0x0f88 },
	{ 61,0,0,1,0x1d78,0x3eb5,0x00000063,0x011580,0x06,0x07f,0x2000 },
	{ 62,0,1,1,0x1df7,0x3eb5,0x00000063,0x000000,0x06,0x07f,0x2000 },
	{ 63,0,1,1,0x1e76,0x3eb5,0x00000063,0x000000,0x06,0x07f,0x2000 },
	{ 64,0,1,1,0x1ef5,0x3eb5,0x00000063,0x000000,0x06,0x07f,0x2000 },
	{ 65,0,1,1,0x1f74,0x3eb5,0x00000063,0x000000,0x06,0x07f,0x2000 },
	{ 66,0,1,1,0x1ff3,0x3eb5,0x00000063,0x000000,0x06,0x07f,0x2000 },
	{ 67,0,1,1,0x0072,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 68,0,1,1,0x00f1,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 69,0,1,1,0x0170,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 70,0,1,1,0x01ef,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 71,0,1,1,0x026e,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 72,0,1,1,0x02ed,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 73,0,1,1,0x036c,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 74,0,1,1,0x03eb,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 75,0,1,1,0x046a,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 76,0,1,1,0x04e9,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 77,0,1,1,0x0568,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 78,0,1,1,0x05e7,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 79,0,1,1,0x0666,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 80,0,1,1,0x06e5,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 81,0,1,1,0x0764,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 82,0,1,1,0x07e3,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 83,0,1,1,0x0862,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 84,0,1,1,0x08e1,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 85,0,1,1,0x0960,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 86,0,1,1,0x09df,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 87,0,1,1,0x0a5e,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 88,0,1,1,0x0add,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 89,0,1,1,0x0b5c,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 90,0,1,1,0x0bdb,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 91,0,1,1,0x0c5a,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 92,0,1,1,0x0cd9,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 93,0,1,1,0x0d58,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 94,0,1,1,0x0dd7,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 95,0,1,1,0x0e56,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 96,0,1,1,0x0ed5,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 97,0,1,1,0x0f54,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 98,0,1,1,0x0fd3,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 99,0,1,1,0x1052,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 100,0,1,1,0x10d1,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 101,0,1,1,0x1150,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 102,0,1,1,0x11cf,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 103,0,1,1,0x124e,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 104,0,1,1,0x12cd,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 105,0,1,1,0x134c,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 106,0,1,1,0x13cb,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 107,0,1,1,0x144a,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 108,0,1,1,0x14c9,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 109,0,1,1,0x1548,0x1eb5,0x000063d4,0x000000,0x06,0x07f,0x1570 },
	{ 110,0,0,1,0x23b5,0x2ca3,0x000005b4,0x005e20,0x07,0x07f,0x2000 },
	{ 111,0,1,1,0x0434,0x0ca3,0x0005b469,0x000000,0x07,0x07f,0x05f5 },
	{ 112,0,1,1,0x04b3,0x0ca3,0x0005b469,0x000000,0x07,0x07f,0x05f5 },
	{ 113,0,1,1,0x0532,0x0ca3,0x0005b469,0x000000,0x07,0x07f,0x05f5 },
	{ 114,0,1,1,0x05b1,0x0ca3,0x0005b469,0x000000,0x07,0x07f,0x05f5 },
	{ 115,0,0,1,0x141e,0x1a91,0x000039a9,0x057ac0,0x08,0x07f,0x189b },
	{ 116,0,1,1,0x149d,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 117,0,1,1,0x151c,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 118,0,1,1,0x159b,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 119,0,1,1,0x161a,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 120,0,1,1,0x1699,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 121,0,1,1,0x1718,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 122,0,1,1,0x1797,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 123,0,1,1,0x1816,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 124,0,1,1,0x1895,0x1a91,0x000039a9,0x000000,0x08,0x07f,0x189b },
	{ 125,0,0,1,0x2702,0x287f,0x0000048b,0x00351e,0x09,0x07f,0x2000 },
	{ 126,0,0,1,0x156f,0x166d,0x00002e6b,0x045cc0,0x0a,0x07f,0x19dc },
	{ 127,0,1,1,0x15ee,0x166d,0x00002e6b,0x000000,0x0a,0x07f,0x19dc },
	{ 128,0,1,1,0x166d,0x468c,0x00002e6b,0x000000,0x00,0x07f,0x19dc },
	{ 129,0,1,1,0x16ec,0x468c,0x00002e6b,0x000000,0x00,0x07f,0x19dc },
	{ 130,0,1,1,0x176b,0x468c,0x00002e6b,0x000000,0x00,0x07f,0x19dc },
	{ 131,0,1,1,0x17ea,0x468c,0x00002e6b,0x000000,0x00,0x07f,0x19dc },
	{ 132,0,1,1,0x1869,0x468c,0x00002e6b,0x000000,0x00,0x07f,0x19dc },
	{ 133,0,1,1,0x18e8,0x468c,0x00002e6b,0x000000,0x00,0x07f,0x19dc },
	{ 134,0,1,1,0x1967,0x468c,0x00002e6b,0x000000,0x00,0x07f,0x19dc },
	{ 135,0,0,1,0x27d4,0x547a,0x00000055,0x002e16,0x01,0x07f,0x2000 },
	{ 136,0,1,1,0x0853,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 137,0,1,1,0x08d2,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 138,0,1,1,0x0951,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 139,0,1,1,0x09d0,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 140,0,1,1,0x0a4f,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 141,0,1,1,0x0ace,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 142,0,1,1,0x0b4d,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 143,0,1,1,0x0bcc,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 144,0,1,1,0x0c4b,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 145,0,1,1,0x0cca,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 146,0,1,1,0x0d49,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 147,0,1,1,0x0dc8,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 148,0,1,1,0x0e47,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 149,0,1,1,0x0ec6,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 150,0,1,1,0x0f45,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 151,0,1,1,0x0fc4,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 152,0,1,1,0x1043,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 153,0,1,1,0x10c2,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 154,0,1,1,0x1141,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 155,0,1,1,0x11c0,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 156,0,1,1,0x123f,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 157,0,1,1,0x12be,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 158,0,1,1,0x133d,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 159,0,1,1,0x13bc,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 160,0,1,1,0x143b,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 161,0,1,1,0x14ba,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 162,0,1,1,0x1539,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 163,0,1,1,0x15b8,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 164,0,1,1,0x1637,0x347a,0x000055f6,0x000000,0x01,0x07f,0x164e },
	{ 165,0,0,1,0x24a4,0x4268,0x000005e2,0x005014,0x02,0x07f,0x2000 },
	{ 166,0,1,1,0x0523,0x2268,0x0005e20a,0x000000,0x02,0x07f,0x05c7 },
	{ 167,0,1,1,0x05a2,0x2268,0x0005e20a,0x000000,0x02,0x07f,0x05c7 },
	{ 168,0,0,1,0x140f,0x3056,0x0000590a,0x058900,0x03,0x07f,0x161a },
	{ 169,0,1,1,0x148e,0x3056,0x0000590a,0x000000,0x03,0x07f,0x161a },
	{ 170,0,1,1,0x150d,0x3056,0x0000590a,0x000000,0x03,0x07f,0x161a },
	{ 171,0,1,1,0x158c,0x3056,0x0000590a,0x000000,0x03,0x07f,0x161a },
	{ 172,0,1,1,0x160b,0x3056,0x0000590a,0x000000,0x03,0x07f,0x161a },
	{ 173,0,0,1,0x2478,0x3e44,0x0000068e,0x00527c,0x04,0x07f,0x2000 },
	{ 174,0,1,1,0x04f7,0x1e44,0x00068eea,0x000000,0x04,0x07f,0x0527 },
	{ 175,0,0,1,0x1364,0x2c32,0x0000582a,0x0636c0,0x05,0x07f,0x1628 },
	{ 176,0,1,1,0x13e3,0x2c32,0x0000582a,0x000000,0x05,0x07f,0x1628 },
	{ 177,0,1,1,0x1462,0x2c32,0x0000582a,0x000000,0x05,0x07f,0x1628 },
	{ 178,0,1,1,0x14e1,0x2c32,0x0000582a,0x000000,0x05,0x07f,0x1628 },
	{ 179,0,1,1,0x1560,0x2c32,0x0000582a,0x000000,0x05,0x07f,0x1628 },
	{ 180,0,1,1,0x15df,0x2c32,0x0000582a,0x000000,0x05,0x07f,0x1628 },
	{ 181,0,0,1,0x244c,0x3a20,0x0000032e,0x0054fc,0x06,0x07f,0x2000 },
	{ 182,0,1,1,0x04cb,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 183,0,1,1,0x054a,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 184,0,1,1,0x05c9,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 185,0,1,1,0x0648,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 186,0,1,1,0x06c7,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 187,0,1,1,0x0746,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 188,0,1,1,0x07c5,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 189,0,1,1,0x0844,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 190,0,1,1,0x08c3,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 191,0,1,1,0x0942,0x1a20,0x00032edf,0x000000,0x06,0x07f,0x0953 },
	{ 192,0,0,1,0x17af,0x280e,0x00003adf,0x02f400,0x07,0x07f,0x187c },
	{ 193,0,1,1,0x182e,0x280e,0x00003adf,0x000000,0x07,0x07f,0x187c },
	{ 194,0,0,1,0x269b,0x35fc,0x000001ed,0x0038f2,0x08,0x07f,0x2000 },
	{ 195,0,1,1,0x071a,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 196,0,1,1,0x0799,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 197,0,1,1,0x0818,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 198,0,1,1,0x0897,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 199,0,1,1,0x0916,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 200,0,1,1,0x0995,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 201,0,1,1,0x0a14,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 202,0,1,1,0x0a93,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 203,0,1,1,0x0b12,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 204,0,1,1,0x0b91,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 205,0,1,1,0x0c10,0x15fc,0x0001ed0a,0x000000,0x08,0x07f,0x0c39 },
	{ 206,0,0,1,0x1a7d,0x23ea,0x00001c2a,0x01d0e0,0x09,0x07f,0x1cbe },
	{ 207,0,1,1,0x1afc,0x23ea,0x00001c2a,0x000000,0x09,0x07f,0x1cbe },
	{ 208,0,1,1,0x1b7b,0x23ea,0x00001c2a,0x000000,0x09,0x07f,0x1cbe },
	{ 209,0,1,1,0x1bfa,0x23ea,0x00001c2a,0x000000,0x09,0x07f,0x1cbe },
	{ 210,0,1,1,0x1c79,0x23ea,0x00001c2a,0x000000,0x09,0x07f,0x1cbe },
	{ 211,0,0,1,0x2ae6,0x31d8,0x00000119,0x001b11,0x0a,0x07f,0x2000 },
	{ 212,0,1,1,0x0b65,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 213,0,1,1,0x0be4,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 214,0,1,1,0x0c63,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 215,0,1,1,0x0ce2,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 216,0,1,1,0x0d61,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 217,0,1,1,0x0de0,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 218,0,1,1,0x0e5f,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 219,0,1,1,0x0ede,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 220,0,1,1,0x0f5d,0x11d8,0x0001194e,0x000000,0x0a,0x07f,0x0f76 },
	{ 221,0,0,1,0x1dca,0x4e63,0x000012de,0x010670,0x00,0x08f,0x1f0e },
	{ 222,0,1,1,0x1e59,0x4e63,0x000012de,0x000000,0x00,0x08f,0x1f0e },
	{ 223,0,1,1,0x1ee8,0x4e63,0x000012de,0x000000,0x00,0x08f,0x1f0e },
	{ 224,0,0,1,0x2cad,0x5b99,0x0000014d,0x001191,0x01,0x08f,0x2000 },
	{ 225,0,1,1,0x0d3c,0x3b99,0x00014d9f,0x000000,0x01,0x08f,0x0e7a },
	{ 226,0,1,1,0x0dcb,0x3b99,0x00014d9f,0x000000,0x01,0x08f,0x0e7a },
	{ 227,0,1,1,0x0e5a,0x3b99,0x00014d9f,0x000000,0x01,0x08f,0x0e7a },
	{ 228,0,0,1,0x1c1f,0x48cf,0x0000182f,0x013570,0x02,0x08f,0x1d9f },
	{ 229,0,1,1,0x1cae,0x48cf,0x0000182f,0x000000,0x02,0x08f,0x1d9f },
	{ 230,0,1,1,0x1d3d,0x48cf,0x0000182f,0x000000,0x02,0x08f,0x1d9f },
	{ 231,0,0,1,0x2b02,0x5605,0x000000bc,0x001773,0x03,0x08f,0x2000 },
	{ 232,0,1,1,0x0b91,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 233,0,1,1,0x0c20,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 234,0,1,1,0x0caf,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 235,0,1,1,0x0d3e,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 236,0,1,1,0x0dcd,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 237,0,1,1,0x0e5c,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 238,0,1,1,0x0eeb,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 239,0,1,1,0x0f7a,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 240,0,1,1,0x1009,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 241,0,1,1,0x1098,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 242,0,1,1,0x1127,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 243,0,1,1,0x11b6,0x3605,0x0000bcc5,0x000000,0x03,0x08f,0x11c3 },
	{ 244,0,0,1,0x1f7b,0x433b,0x00000fe5,0x00ace0,0x04,0x08f,0x2000 },
	{ 245,0,0,1,0x0d40,0x3071,0x00000000,0x0fe580,0x05,0x08f,0x2000 },
	{ 246,0,1,1,0x0dcf,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 247,0,1,1,0x0e5e,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 248,0,1,1,0x0eed,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 249,0,1,1,0x0f7c,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 250,0,1,1,0x100b,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 251,0,1,1,0x109a,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 252,0,1,1,0x1129,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 253,0,1,1,0x11b8,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 254,0,1,1,0x1247,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 255,0,1,1,0x12d6,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
	{ 256,0,1,1,0x1365,0x3071,0x00000000,0x000000,0x05,0x08f,0x2000 },
};

uint8_t stream1[]={ 0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x01, };

struct KnownState knownstates1[]=
{
//	 i,s,yn,mps,  lr,   lrm,         x,      dx,   k,   lp,    lx
	{ 0,-1,-1,-1,0x1001,0x1001,0x00000001,-1,-1,-1,-1 },
	{ 1,0,0,0,0x1401,0x5001,0x00000001,0x000000,0x00,0x400,0x2000 },
	{ 2,0,0,0,0x1801,0x5001,0x00000001,0x000000,0x00,0x400,0x2000 },
	{ 3,0,0,0,0x1c01,0x5001,0x00000001,0x000000,0x00,0x400,0x2000 },
	{ 4,0,0,0,0x0001,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 5,0,0,0,0x0401,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 6,0,0,0,0x0801,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 7,0,0,0,0x0c01,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 8,0,0,0,0x1001,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 9,0,0,0,0x1401,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 10,0,0,0,0x1801,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 11,0,0,0,0x1c01,0x3001,0x00000100,0x000000,0x00,0x400,0x2000 },
	{ 12,0,0,0,0x0001,0x1001,0x00010001,0x000000,0x00,0x400,0x1000 },
	{ 13,0,0,0,0x0401,0x1001,0x00010001,0x000000,0x00,0x400,0x1000 },
	{ 14,0,0,0,0x0801,0x1001,0x00010001,0x000000,0x00,0x400,0x1000 },
	{ 15,0,0,0,0x0c01,0x1001,0x00010001,0x000000,0x00,0x400,0x1000 },
	{ 16,0,1,0,0x1001,0x1001,0x00000019,0x00ffe8,0x01,0x400,0x2000 },
	{ 17,0,0,0,0x1401,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 18,0,0,0,0x154c,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 19,0,0,0,0x1697,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 20,0,0,0,0x17e2,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 21,0,0,0,0x192d,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 22,0,0,0,0x1a78,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 23,0,0,0,0x1bc3,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 24,0,0,0,0x1d0e,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 25,0,0,0,0x1e59,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 26,0,0,0,0x1fa4,0x47b9,0x00000019,0x000000,0x00,0x14b,0x2000 },
	{ 27,0,0,0,0x00ef,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 28,0,0,0,0x023a,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 29,0,0,0,0x0385,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 30,0,0,0,0x04d0,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 31,0,0,0,0x061b,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 32,0,0,0,0x0766,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 33,0,0,0,0x08b1,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 34,0,0,0,0x09fc,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 35,0,0,0,0x0b47,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 36,0,0,0,0x0c92,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 37,0,0,0,0x0ddd,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 38,0,0,0,0x0f28,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 39,0,0,0,0x1073,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 40,0,0,0,0x11be,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 41,0,0,0,0x1309,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 42,0,0,0,0x1454,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 43,0,0,0,0x159f,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 44,0,0,0,0x16ea,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 45,0,0,0,0x1835,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 46,0,0,0,0x1980,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 47,0,0,0,0x1acb,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 48,0,0,0,0x1c16,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 49,0,0,0,0x1d61,0x27b9,0x00001900,0x000000,0x00,0x14b,0x1d6e },
	{ 50,0,1,0,0x26a8,0x2fb5,0x000004d9,0x001427,0x01,0x14b,0x2000 },
	{ 51,0,1,0,0x0fef,0x17b1,0x0000cf81,0x040980,0x02,0x14b,0x1137 },
	{ 52,0,1,0,0x1936,0x1fad,0x00000071,0x00cf10,0x03,0x14b,0x2000 },
	{ 53,0,0,0,0x1a81,0x1fad,0x00000071,0x000000,0x03,0x14b,0x2000 },
	{ 54,0,0,0,0x1bcc,0x1fad,0x00000071,0x000000,0x03,0x14b,0x2000 },
	{ 55,0,0,0,0x1d17,0x1fad,0x00000071,0x000000,0x03,0x14b,0x2000 },
	{ 56,0,0,0,0x1e62,0x1fad,0x00000071,0x000000,0x03,0x14b,0x2000 },
	{ 57,0,0,0,0x1fad,0x5220,0x00000071,0x000000,0x00,0x11f,0x2000 },
	{ 58,0,0,0,0x00cc,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 59,0,0,0,0x01eb,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 60,0,0,0,0x030a,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 61,0,0,0,0x0429,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 62,0,0,0,0x0548,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 63,0,0,0,0x0667,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 64,0,0,0,0x0786,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 65,0,0,0,0x08a5,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 66,0,0,0,0x09c4,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 67,0,0,0,0x0ae3,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 68,0,0,0,0x0c02,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 69,0,0,0,0x0d21,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 70,0,0,0,0x0e40,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 71,0,0,0,0x0f5f,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 72,0,0,0,0x107e,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 73,0,0,0,0x119d,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 74,0,0,0,0x12bc,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 75,0,0,0,0x13db,0x3220,0x00007100,0x000000,0x00,0x11f,0x14b9 },
	{ 76,0,1,0,0x1de0,0x3b06,0x000004e4,0x006c1c,0x01,0x11f,0x2000 },
	{ 77,0,0,0,0x1eff,0x3b06,0x000004e4,0x000000,0x01,0x11f,0x2000 },
	{ 78,0,0,0,0x001e,0x1b06,0x0004e401,0x000000,0x01,0x11f,0x06d8 },
	{ 79,0,0,0,0x013d,0x1b06,0x0004e401,0x000000,0x01,0x11f,0x06d8 },
	{ 80,0,0,0,0x025c,0x1b06,0x0004e401,0x000000,0x01,0x11f,0x06d8 },
	{ 81,0,0,0,0x037b,0x1b06,0x0004e401,0x000000,0x01,0x11f,0x06d8 },
	{ 82,0,0,0,0x049a,0x1b06,0x0004e401,0x000000,0x01,0x11f,0x06d8 },
	{ 83,0,0,0,0x05b9,0x1b06,0x0004e401,0x000000,0x01,0x11f,0x06d8 },
	{ 84,0,1,0,0x0fbe,0x23ec,0x00000001,0x04e400,0x02,0x11f,0x2000 },
	{ 85,0,0,0,0x10dd,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 86,0,0,0,0x11fc,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 87,0,0,0,0x131b,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 88,0,0,0,0x143a,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 89,0,0,0,0x1559,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 90,0,0,0,0x1678,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 91,0,0,0,0x1797,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 92,0,0,0,0x18b6,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 93,0,0,0,0x19d5,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 94,0,0,0,0x1af4,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 95,0,0,0,0x1c13,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 96,0,0,0,0x1d32,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 97,0,0,0,0x1e51,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 98,0,0,0,0x1f70,0x23ec,0x00000001,0x000000,0x02,0x11f,0x2000 },
	{ 99,0,0,0,0x008f,0x03ec,0x00000100,0x000000,0x02,0x11f,0x2000 },
	{ 100,0,0,0,0x01ae,0x03ec,0x00000100,0x000000,0x02,0x11f,0x2000 },
	{ 101,0,0,0,0x02cd,0x03ec,0x00000100,0x000000,0x02,0x11f,0x2000 },
	{ 102,0,0,0,0x03ec,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 103,0,0,0,0x04e3,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 104,0,0,0,0x05da,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 105,0,0,0,0x06d1,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 106,0,0,0,0x07c8,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 107,0,0,0,0x08bf,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 108,0,0,0,0x09b6,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 109,0,0,0,0x0aad,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 110,0,0,0,0x0ba4,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 111,0,0,0,0x0c9b,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 112,0,0,0,0x0d92,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 113,0,0,0,0x0e89,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 114,0,0,0,0x0f80,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 115,0,0,0,0x1077,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 116,0,0,0,0x116e,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 117,0,0,0,0x1265,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 118,0,0,0,0x135c,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 119,0,0,0,0x1453,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 120,0,0,0,0x154a,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 121,0,0,0,0x1641,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 122,0,0,0,0x1738,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 123,0,0,0,0x182f,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 124,0,0,0,0x1926,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 125,0,0,0,0x1a1d,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 126,0,0,0,0x1b14,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 127,0,0,0,0x1c0b,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 128,0,0,0,0x1d02,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 129,0,0,0,0x1df9,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 130,0,0,0,0x1ef0,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 131,0,0,0,0x1fe7,0x3618,0x00000100,0x000000,0x00,0x0f7,0x2000 },
	{ 132,0,0,0,0x00de,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 133,0,0,0,0x01d5,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 134,0,0,0,0x02cc,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 135,0,0,0,0x03c3,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 136,0,0,0,0x04ba,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 137,0,0,0,0x05b1,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 138,0,0,0,0x06a8,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 139,0,0,0,0x079f,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 140,0,0,0,0x0896,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 141,0,0,0,0x098d,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 142,0,0,0,0x0a84,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 143,0,0,0,0x0b7b,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 144,0,0,0,0x0c72,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 145,0,0,0,0x0d69,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 146,0,0,0,0x0e60,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 147,0,0,0,0x0f57,0x1618,0x00010001,0x000000,0x00,0x0f7,0x1000 },
	{ 148,0,1,0,0x1a27,0x1ff1,0x00000d19,0x00f2e8,0x01,0x0f7,0x2000 },
	{ 149,0,0,0,0x1b1e,0x1ff1,0x00000d19,0x000000,0x01,0x0f7,0x2000 },
	{ 150,0,0,0,0x1c15,0x1ff1,0x00000d19,0x000000,0x01,0x0f7,0x2000 },
	{ 151,0,0,0,0x1d0c,0x1ff1,0x00000d19,0x000000,0x01,0x0f7,0x2000 },
	{ 152,0,0,0,0x1e03,0x1ff1,0x00000d19,0x000000,0x01,0x0f7,0x2000 },
	{ 153,0,0,0,0x1efa,0x1ff1,0x00000d19,0x000000,0x01,0x0f7,0x2000 },
	{ 154,0,0,0,0x1ff1,0x4fcb,0x00000d19,0x000000,0x00,0x062,0x2000 },
	{ 155,0,0,0,0x0053,0x2fcb,0x000d1900,0x000000,0x00,0x062,0x0129 },
	{ 156,0,0,0,0x00b5,0x2fcb,0x000d1900,0x000000,0x00,0x062,0x0129 },
	{ 157,0,0,0,0x0117,0x2fcb,0x000d1900,0x000000,0x00,0x062,0x0129 },
	{ 158,0,1,0,0x10f6,0x3f48,0x0000b200,0x0c6700,0x01,0x062,0x121a },
	{ 159,0,0,0,0x1158,0x3f48,0x0000b200,0x000000,0x01,0x062,0x121a },
	{ 160,0,0,0,0x11ba,0x3f48,0x0000b200,0x000000,0x01,0x062,0x121a },
	{ 161,0,1,0,0x2199,0x4ec5,0x00000048,0x00b1b8,0x02,0x062,0x2000 },
	{ 162,0,0,0,0x01fb,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 163,0,0,0,0x025d,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 164,0,0,0,0x02bf,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 165,0,0,0,0x0321,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 166,0,0,0,0x0383,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 167,0,0,0,0x03e5,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 168,0,0,0,0x0447,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 169,0,0,0,0x04a9,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 170,0,0,0,0x050b,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 171,0,0,0,0x056d,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 172,0,0,0,0x05cf,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 173,0,0,0,0x0631,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 174,0,0,0,0x0693,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 175,0,0,0,0x06f5,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 176,0,0,0,0x0757,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 177,0,0,0,0x07b9,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 178,0,0,0,0x081b,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 179,0,0,0,0x087d,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 180,0,0,0,0x08df,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 181,0,0,0,0x0941,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 182,0,0,0,0x09a3,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 183,0,0,0,0x0a05,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 184,0,0,0,0x0a67,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 185,0,0,0,0x0ac9,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 186,0,0,0,0x0b2b,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 187,0,0,0,0x0b8d,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 188,0,0,0,0x0bef,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 189,0,0,0,0x0c51,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 190,0,0,0,0x0cb3,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 191,0,0,0,0x0d15,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 192,0,0,0,0x0d77,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 193,0,0,0,0x0dd9,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 194,0,0,0,0x0e3b,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 195,0,0,0,0x0e9d,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 196,0,0,0,0x0eff,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 197,0,0,0,0x0f61,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 198,0,0,0,0x0fc3,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 199,0,0,0,0x1025,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 200,0,0,0,0x1087,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 201,0,0,0,0x10e9,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 202,0,0,0,0x114b,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 203,0,0,0,0x11ad,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 204,0,0,0,0x120f,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 205,0,0,0,0x1271,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 206,0,0,0,0x12d3,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 207,0,0,0,0x1335,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 208,0,0,0,0x1397,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 209,0,0,0,0x13f9,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 210,0,0,0,0x145b,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 211,0,0,0,0x14bd,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 212,0,0,0,0x151f,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 213,0,0,0,0x1581,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 214,0,0,0,0x15e3,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 215,0,0,0,0x1645,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 216,0,0,0,0x16a7,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 217,0,0,0,0x1709,0x2ec5,0x00004801,0x000000,0x02,0x062,0x1753 },
	{ 218,0,1,0,0x26e8,0x3e42,0x00000129,0x0046d8,0x03,0x062,0x2000 },
	{ 219,0,0,0,0x074a,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 220,0,0,0,0x07ac,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 221,0,0,0,0x080e,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 222,0,0,0,0x0870,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 223,0,0,0,0x08d2,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 224,0,0,0,0x0934,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 225,0,0,0,0x0996,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 226,0,0,0,0x09f8,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 227,0,0,0,0x0a5a,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 228,0,0,0,0x0abc,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 229,0,0,0,0x0b1e,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 230,0,0,0,0x0b80,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 231,0,0,0,0x0be2,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 232,0,0,0,0x0c44,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 233,0,0,0,0x0ca6,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 234,0,0,0,0x0d08,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 235,0,0,0,0x0d6a,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 236,0,0,0,0x0dcc,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 237,0,0,0,0x0e2e,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 238,0,0,0,0x0e90,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 239,0,0,0,0x0ef2,0x1e42,0x00012900,0x000000,0x03,0x062,0x0f26 },
	{ 240,0,1,0,0x1ed1,0x2dbf,0x00000930,0x011fd0,0x04,0x062,0x2000 },
	{ 241,0,0,0,0x1f33,0x2dbf,0x00000930,0x000000,0x04,0x062,0x2000 },
	{ 242,0,0,0,0x1f95,0x2dbf,0x00000930,0x000000,0x04,0x062,0x2000 },
	{ 243,0,0,0,0x1ff7,0x2dbf,0x00000930,0x000000,0x04,0x062,0x2000 },
	{ 244,0,0,0,0x0059,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 245,0,0,0,0x00bb,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 246,0,0,0,0x011d,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 247,0,0,0,0x017f,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 248,0,0,0,0x01e1,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 249,0,0,0,0x0243,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 250,0,0,0,0x02a5,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 251,0,0,0,0x0307,0x0dbf,0x00093001,0x000000,0x04,0x062,0x0335 },
	{ 252,0,1,0,0x12e6,0x1d3c,0x00005201,0x08de00,0x05,0x062,0x1693 },
	{ 253,0,0,0,0x1348,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 254,0,0,0,0x13aa,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 255,0,0,0,0x140c,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 256,0,0,0,0x146e,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 257,0,0,0,0x14d0,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 258,0,0,0,0x1532,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 259,0,0,0,0x1594,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 260,0,0,0,0x15f6,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 261,0,0,0,0x1658,0x1d3c,0x00005201,0x000000,0x05,0x062,0x1693 },
	{ 262,0,1,0,0x2637,0x2cb9,0x00000225,0x004fdc,0x06,0x062,0x2000 },
	{ 263,0,0,0,0x0699,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 264,0,0,0,0x06fb,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 265,0,0,0,0x075d,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 266,0,0,0,0x07bf,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 267,0,0,0,0x0821,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 268,0,0,0,0x0883,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 269,0,0,0,0x08e5,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 270,0,0,0,0x0947,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 271,0,0,0,0x09a9,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 272,0,0,0,0x0a0b,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 273,0,0,0,0x0a6d,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 274,0,0,0,0x0acf,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 275,0,0,0,0x0b31,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 276,0,0,0,0x0b93,0x0cb9,0x00022500,0x000000,0x06,0x062,0x0b9a },
	{ 277,0,1,0,0x1b72,0x1c36,0x00002100,0x020400,0x07,0x062,0x1bd4 },
	{ 278,0,1,0,0x2b51,0x2bb3,0x00000004,0x0020fc,0x08,0x062,0x2000 },
	{ 279,0,0,0,0x0bb3,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 280,0,0,0,0x0c15,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 281,0,0,0,0x0c77,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 282,0,0,0,0x0cd9,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 283,0,0,0,0x0d3b,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 284,0,0,0,0x0d9d,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 285,0,0,0,0x0dff,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 286,0,0,0,0x0e61,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 287,0,0,0,0x0ec3,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 288,0,0,0,0x0f25,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 289,0,0,0,0x0f87,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 290,0,0,0,0x0fe9,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 291,0,0,0,0x104b,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 292,0,0,0,0x10ad,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 293,0,0,0,0x110f,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 294,0,0,0,0x1171,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 295,0,0,0,0x11d3,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 296,0,0,0,0x1235,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 297,0,0,0,0x1297,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 298,0,0,0,0x12f9,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 299,0,0,0,0x135b,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 300,0,0,0,0x13bd,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 301,0,0,0,0x141f,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 302,0,0,0,0x1481,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 303,0,0,0,0x14e3,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 304,0,0,0,0x1545,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 305,0,0,0,0x15a7,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 306,0,0,0,0x1609,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 307,0,0,0,0x166b,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 308,0,0,0,0x16cd,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 309,0,0,0,0x172f,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 310,0,0,0,0x1791,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 311,0,0,0,0x17f3,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 312,0,0,0,0x1855,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 313,0,0,0,0x18b7,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 314,0,0,0,0x1919,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 315,0,0,0,0x197b,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 316,0,0,0,0x19dd,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 317,0,0,0,0x1a3f,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 318,0,0,0,0x1aa1,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 319,0,0,0,0x1b03,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 320,0,0,0,0x1b65,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 321,0,0,0,0x1bc7,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 322,0,0,0,0x1c29,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 323,0,0,0,0x1c8b,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 324,0,0,0,0x1ced,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 325,0,0,0,0x1d4f,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 326,0,0,0,0x1db1,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 327,0,0,0,0x1e13,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 328,0,0,0,0x1e75,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 329,0,0,0,0x1ed7,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 330,0,0,0,0x1f39,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 331,0,0,0,0x1f9b,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 332,0,0,0,0x1ffd,0x3b8d,0x00000401,0x000000,0x00,0x062,0x2000 },
	{ 333,0,0,0,0x005f,0x1b8d,0x00040100,0x000000,0x00,0x062,0x07ff },
};

uint8_t stream2[]={ 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff, };

struct KnownState knownstates2[]=
{
//	 i,s,yn,mps,  lr,   lrm,         x,      dx,   k,   lp,    lx
	{ 0,-1,-1,-1,0x1001,0x1001,0x0000ffff,-1,-1,-1,-1 },
	{ 1,0,1,0,0x1401,0x5001,0x0000800b,0x007ff4,0x01,0x400,0x1400 },
	{ 2,0,1,0,0x1801,0x5001,0x00004011,0x003ffa,0x02,0x400,0x17ff },
	{ 3,0,1,0,0x1c01,0x5001,0x00002014,0x001ffd,0x03,0x400,0x1bfd },
	{ 4,0,1,0,0x0001,0x3001,0x0011157f,0x0ffe80,0x04,0x400,0xffa1 },
	{ 5,0,1,0,0x0401,0x3001,0x0009163f,0x07ff40,0x05,0x400,0x0345 },
	{ 6,0,1,0,0x0801,0x3001,0x0005169f,0x03ffa0,0x06,0x400,0x069e },
	{ 7,0,1,0,0x0c01,0x3001,0x000316cf,0x01ffd0,0x07,0x400,0x097f },
	{ 8,0,1,0,0x1001,0x3001,0x000216e7,0x00ffe8,0x08,0x400,0x0bc1 },
	{ 9,0,1,0,0x1401,0x3001,0x000196f3,0x007ff4,0x09,0x400,0x0d54 },
	{ 10,0,1,0,0x1801,0x3001,0x000156f9,0x003ffa,0x0a,0x400,0x0e51 },
	{ 11,0,1,1,0x1c01,0x5702,0x000136fc,0x001ffd,0x00,0x31b,0x0ee2 },
	{ 12,0,0,1,0x2112,0x58f8,0x0001244d,0x0012af,0x01,0x31b,0x0f3e },
	{ 13,0,0,1,0x0623,0x3aee,0x011d89ff,0x07c300,0x02,0x31b,0xfc77 },
	{ 14,0,0,1,0x0b34,0x3ce4,0x011a4fff,0x033a00,0x03,0x31b,0xfd22 },
	{ 15,0,0,1,0x1045,0x3eda,0x0118f89f,0x015760,0x04,0x31b,0xfd70 },
	{ 16,0,0,1,0x1556,0x40d0,0x011869df,0x008ec0,0x05,0x31b,0xfd91 },
	{ 17,0,0,1,0x1a67,0x42c6,0x01182e91,0x003b4e,0x06,0x31b,0xfd9f },
	{ 18,0,0,1,0x1f78,0x44bc,0x011815ea,0x0018a7,0x07,0x31b,0xfda5 },
	{ 19,0,0,1,0x0489,0x26b2,0x180caa7f,0x0a3f80,0x08,0x31b,0xfca4 },
	{ 20,0,0,1,0x099a,0x28a8,0x180867ff,0x044280,0x09,0x31b,0xfd92 },
	{ 21,0,0,1,0x0eab,0x2a9e,0x1806a2df,0x01c520,0x0a,0x31b,0xfe01 },
	{ 22,0,0,0,0x13bc,0x53bc,0x1805e67f,0x00bc60,0x00,0x400,0xfe32 },
	{ 23,0,1,0,0x17bc,0x53bc,0x1805a373,0x00430c,0x01,0x400,0xfe43 },
	{ 24,0,1,0,0x1bbc,0x53bc,0x180581ed,0x002186,0x02,0x400,0xfe4c },
	{ 25,0,1,0,0x1fbc,0x53bc,0x1805712a,0x0010c3,0x03,0x400,0xfe51 },
	{ 26,0,1,0,0x03bc,0x33bc,0x0569c87f,0x086180,0x04,0x400,0xfd40 },
	{ 27,0,1,0,0x07bc,0x33bc,0x056597bf,0x0430c0,0x05,0x400,0xfe47 },
	{ 28,0,1,0,0x0bbc,0x33bc,0x05637f5f,0x021860,0x06,0x400,0xfedd },
	{ 29,0,1,0,0x0fbc,0x33bc,0x0562732f,0x010c30,0x07,0x400,0xff2f },
	{ 30,0,1,0,0x13bc,0x33bc,0x0561ed17,0x008618,0x08,0x400,0xff59 },
	{ 31,0,1,0,0x17bc,0x33bc,0x0561aa0b,0x00430c,0x09,0x400,0xff6f },
	{ 32,0,1,0,0x1bbc,0x33bc,0x05618885,0x002186,0x0a,0x400,0xff7a },
	{ 33,0,1,1,0x1fbc,0x5abd,0x056177c2,0x0010c3,0x00,0x31b,0xff80 },
	{ 34,0,0,1,0x04cd,0x3cb3,0x616ef87f,0x09c980,0x01,0x31b,0xfc31 },
	{ 35,0,0,1,0x09de,0x3ea9,0x616ae73f,0x041140,0x02,0x31b,0xfd01 },
	{ 36,0,0,1,0x0eef,0x409f,0x6169367f,0x01b0c0,0x03,0x31b,0xfd61 },
	{ 37,0,0,1,0x1400,0x4295,0x61688297,0x00b3e8,0x04,0x31b,0xfd8b },
	{ 38,0,0,1,0x1911,0x448b,0x616837cf,0x004ac8,0x05,0x31b,0xfd9d },
	{ 39,0,0,1,0x1e22,0x4681,0x616818bd,0x001f12,0x06,0x31b,0xfda4 },
	{ 40,0,0,1,0x0333,0x2877,0x680cd27f,0x0cea80,0x07,0x31b,0xfc9b },
	{ 41,0,0,1,0x0844,0x2a6d,0x680773ff,0x055e80,0x08,0x31b,0xfdcc },
	{ 42,0,0,1,0x0d55,0x2c63,0x6805389f,0x023b60,0x09,0x31b,0xfe60 },
	{ 43,0,0,1,0x1266,0x2e59,0x68044b3f,0x00ed60,0x0a,0x31b,0xfea2 },
	{ 44,0,0,0,0x1777,0x5777,0x6803e88f,0x0062b0,0x00,0x400,0xfebf },
	{ 45,0,1,0,0x1b77,0x5777,0x6803c56d,0x002322,0x01,0x400,0xfec9 },
	{ 46,0,1,0,0x1f77,0x5777,0x6803b3dc,0x001191,0x02,0x400,0xfece },
	{ 47,0,1,0,0x0377,0x3777,0x03ac137f,0x08c880,0x03,0x400,0xfcc2 },
	{ 48,0,1,0,0x0777,0x3777,0x03a7af3f,0x046440,0x04,0x400,0xfdbe },
	{ 49,0,1,0,0x0b77,0x3777,0x03a57d1f,0x023220,0x05,0x400,0xfe4e },
	{ 50,0,1,0,0x0f77,0x3777,0x03a4640f,0x011910,0x06,0x400,0xfe9b },
	{ 51,0,1,0,0x1377,0x3777,0x03a3d787,0x008c88,0x07,0x400,0xfec4 },
	{ 52,0,1,0,0x1777,0x3777,0x03a39143,0x004644,0x08,0x400,0xfed8 },
	{ 53,0,1,0,0x1b77,0x3777,0x03a36e21,0x002322,0x09,0x400,0xfee2 },
	{ 54,0,1,0,0x1f77,0x3777,0x03a35c90,0x001191,0x0a,0x400,0xfee8 },
	{ 55,0,1,1,0x0377,0x3e78,0xa354c77f,0x08c880,0x00,0x31b,0xfe7f },
	{ 56,0,0,1,0x0888,0x406e,0xa34fa6bf,0x0520c0,0x01,0x31b,0xfc11 },
	{ 57,0,0,1,0x0d99,0x4264,0xa34d851f,0x0221a0,0x02,0x31b,0xfc78 },
	{ 58,0,0,1,0x12aa,0x445a,0xa34ca267,0x00e2b8,0x03,0x31b,0xfca5 },
	{ 59,0,0,1,0x17bb,0x4650,0xa34c4427,0x005e40,0x04,0x31b,0xfcb8 },
	{ 60,0,0,1,0x1ccc,0x4846,0xa34c1cf9,0x00272e,0x05,0x31b,0xfcc0 },
	{ 61,0,0,1,0x21dd,0x4a3c,0xa34c0cb1,0x001048,0x06,0x31b,0xfcc4 },
	{ 62,0,0,1,0x06ee,0x2c32,0x4c06ecbf,0x06c440,0x07,0x31b,0xfdee },
	{ 63,0,0,1,0x0bff,0x2e28,0x4c041c9f,0x02d020,0x08,0x31b,0xfeb0 },
	{ 64,0,0,1,0x1110,0x301e,0x4c02f14f,0x012b50,0x09,0x31b,0xff08 },
	{ 65,0,0,1,0x1621,0x3214,0x4c0274f3,0x007c5c,0x0a,0x31b,0xff2f },
	{ 66,0,0,0,0x1b32,0x5b32,0x4c024141,0x0033b2,0x00,0x400,0xff3f },
	{ 67,0,1,0,0x1f32,0x5b32,0x4c022ed9,0x001268,0x01,0x400,0xff45 },
	{ 68,0,1,0,0x0332,0x3b32,0x0226a4ff,0x093400,0x02,0x400,0xfe00 },
};

uint8_t stream3[]={ 0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef, };

struct KnownState knownstates3[]=
{
//	 i,s,yn,mps,  lr,   lrm,         x,      dx,   k,   lp,    lx
	{ 0,-1,-1,-1,0x1001,0x1001,0x00000123,-1,-1,-1,-1 },
	{ 1,0,0,0,0x1401,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 2,0,0,0,0x1801,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 3,0,0,0,0x1c01,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 4,0,0,0,0x0001,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 5,0,0,0,0x0401,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 6,0,0,0,0x0801,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 7,0,0,0,0x0c01,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 8,0,1,0,0x1001,0x3001,0x0000235d,0x00ffe8,0x01,0x400,0x1b6e },
	{ 9,0,0,0,0x1401,0x3001,0x0000235d,0x000000,0x01,0x400,0x1b6e },
	{ 10,0,0,0,0x1801,0x3001,0x0000235d,0x000000,0x01,0x400,0x1b6e },
	{ 11,0,1,0,0x1c01,0x3001,0x00000360,0x001ffd,0x02,0x400,0x2000 },
	{ 12,0,0,0,0x0001,0x1001,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 13,0,0,0,0x0401,0x1001,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 14,0,0,0,0x0801,0x1001,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 15,0,1,0,0x0c01,0x1001,0x00016097,0x01ffd0,0x03,0x400,0x0e28 },
	{ 16,0,1,0,0x1001,0x1001,0x000060af,0x00ffe8,0x04,0x400,0x15a0 },
	{ 17,0,0,0,0x1401,0x52ef,0x000060af,0x000000,0x00,0x37f,0x15a0 },
	{ 18,0,1,0,0x1890,0x53ff,0x00001ad7,0x0045d8,0x01,0x37f,0x1d05 },
	{ 19,0,0,0,0x1c0f,0x53ff,0x00001ad7,0x000000,0x01,0x37f,0x1d05 },
	{ 20,0,1,0,0x209e,0x550f,0x0000098b,0x00114c,0x02,0x37f,0x2000 },
	{ 21,0,1,0,0x052d,0x361f,0x0001b2c9,0x07d8c0,0x03,0x37f,0x0cf3 },
	{ 22,0,0,0,0x08ac,0x361f,0x0001b2c9,0x000000,0x03,0x37f,0x0cf3 },
	{ 23,0,0,0,0x0c2b,0x361f,0x0001b2c9,0x000000,0x03,0x37f,0x0cf3 },
	{ 24,0,1,0,0x10ba,0x372f,0x0000a349,0x010f80,0x04,0x37f,0x129a },
	{ 25,0,1,0,0x1549,0x383f,0x00002819,0x007b30,0x05,0x37f,0x1ab4 },
	{ 26,0,0,0,0x18c8,0x383f,0x00002819,0x000000,0x05,0x37f,0x1ab4 },
	{ 27,0,1,0,0x1d57,0x394f,0x00000997,0x001e82,0x06,0x37f,0x2000 },
	{ 28,0,0,0,0x00d6,0x194f,0x000997ab,0x000000,0x06,0x37f,0x02f5 },
	{ 29,0,1,0,0x0565,0x1a5f,0x000209ab,0x078e00,0x07,0x37f,0x0be5 },
	{ 30,0,0,0,0x08e4,0x1a5f,0x000209ab,0x000000,0x07,0x37f,0x0be5 },
	{ 31,0,1,0,0x0d73,0x1b6f,0x00002abb,0x01def0,0x08,0x37f,0x1a56 },
	{ 32,0,0,0,0x10f2,0x1b6f,0x00002abb,0x000000,0x08,0x37f,0x1a56 },
	{ 33,0,0,0,0x1471,0x1b6f,0x00002abb,0x000000,0x08,0x37f,0x1a56 },
	{ 34,0,0,0,0x17f0,0x1b6f,0x00002abb,0x000000,0x08,0x37f,0x1a56 },
	{ 35,0,1,0,0x1c7f,0x1c7f,0x00000769,0x002352,0x09,0x37f,0x2000 },
	{ 36,0,0,0,0x1ffe,0x5eec,0x00000769,0x000000,0x00,0x37f,0x2000 },
	{ 37,0,0,0,0x037d,0x3eec,0x000769cd,0x000000,0x00,0x37f,0x0471 },
	{ 38,0,1,0,0x080c,0x3ffc,0x0002a40d,0x04c5c0,0x01,0x37f,0x0a67 },
	{ 39,0,1,0,0x0c9b,0x410c,0x0000796d,0x022aa0,0x02,0x37f,0x144f },
	{ 40,0,0,0,0x101a,0x410c,0x0000796d,0x000000,0x02,0x37f,0x144f },
	{ 41,0,0,0,0x1399,0x410c,0x0000796d,0x000000,0x02,0x37f,0x144f },
	{ 42,0,1,0,0x1828,0x421c,0x00002e7d,0x004af0,0x03,0x37f,0x19da },
	{ 43,0,1,0,0x1cb7,0x432c,0x00000c7b,0x002202,0x04,0x37f,0x2000 },
	{ 44,0,0,0,0x0036,0x232c,0x000c7bef,0x000000,0x04,0x37f,0x0170 },
	{ 45,0,1,0,0x04c5,0x243c,0x00040fef,0x086c00,0x05,0x37f,0x07ea },
	{ 46,0,1,0,0x0954,0x254c,0x00003daf,0x03d240,0x06,0x37f,0x1837 },
	{ 47,0,0,0,0x0cd3,0x254c,0x00003daf,0x000000,0x06,0x37f,0x1837 },
	{ 48,0,0,0,0x1052,0x254c,0x00003daf,0x000000,0x06,0x37f,0x1837 },
	{ 49,0,0,0,0x13d1,0x254c,0x00003daf,0x000000,0x06,0x37f,0x1837 },
	{ 50,0,0,0,0x1750,0x254c,0x00003daf,0x000000,0x06,0x37f,0x1837 },
	{ 51,0,1,0,0x1bdf,0x265c,0x00001653,0x00275c,0x07,0x37f,0x1e15 },
	{ 52,0,1,0,0x206e,0x276c,0x00000475,0x0011de,0x08,0x37f,0x2000 },
	{ 53,0,0,0,0x03ed,0x076c,0x00047501,0x000000,0x08,0x37f,0x0762 },
	{ 54,0,1,0,0x087c,0x087c,0x00000841,0x046cc0,0x09,0x37f,0x2000 },
	{ 55,0,0,0,0x0bfb,0x4ae9,0x00000841,0x000000,0x00,0x37f,0x2000 },
	{ 56,0,0,0,0x0f7a,0x4ae9,0x00000841,0x000000,0x00,0x37f,0x2000 },
	{ 57,0,0,0,0x12f9,0x4ae9,0x00000841,0x000000,0x00,0x37f,0x2000 },
	{ 58,0,0,0,0x1678,0x4ae9,0x00000841,0x000000,0x00,0x37f,0x2000 },
	{ 59,0,0,0,0x19f7,0x4ae9,0x00000841,0x000000,0x00,0x37f,0x2000 },
	{ 60,0,0,0,0x1d76,0x4ae9,0x00000841,0x000000,0x00,0x37f,0x2000 },
	{ 61,0,0,0,0x00f5,0x2ae9,0x00084123,0x000000,0x00,0x37f,0x03d3 },
	{ 62,0,1,0,0x0584,0x2bf9,0x0000db23,0x076600,0x01,0x37f,0x10e7 },
	{ 63,0,0,0,0x0903,0x2bf9,0x0000db23,0x000000,0x01,0x37f,0x10e7 },
	{ 64,0,0,0,0x0c82,0x2bf9,0x0000db23,0x000000,0x01,0x37f,0x10e7 },
	{ 65,0,0,0,0x1001,0x2bf9,0x0000db23,0x000000,0x01,0x37f,0x10e7 },
	{ 66,0,1,0,0x1490,0x2d09,0x00004f73,0x008bb0,0x02,0x37f,0x16c2 },
	{ 67,0,1,0,0x191f,0x2e19,0x00001015,0x003f5e,0x03,0x37f,0x1ff9 },
	{ 68,0,0,0,0x1c9e,0x2e19,0x00001015,0x000000,0x03,0x37f,0x1ff9 },
	{ 69,0,1,0,0x012d,0x0f29,0x000063c5,0x0fb180,0x04,0x37f,0x1571 },
	{ 70,0,0,0,0x04ac,0x0f29,0x000063c5,0x000000,0x04,0x37f,0x1571 },
	{ 71,0,0,0,0x082b,0x0f29,0x000063c5,0x000000,0x04,0x37f,0x1571 },
	{ 72,0,0,0,0x0baa,0x0f29,0x000063c5,0x000000,0x04,0x37f,0x1571 },
	{ 73,0,0,0,0x0f29,0x4a2a,0x000063c5,0x000000,0x00,0x31b,0x1571 },
	{ 74,0,0,0,0x1244,0x4a2a,0x000063c5,0x000000,0x00,0x31b,0x1571 },
	{ 75,0,0,0,0x155f,0x4a2a,0x000063c5,0x000000,0x00,0x31b,0x1571 },
	{ 76,0,1,0,0x1a70,0x4c20,0x000028d3,0x003af2,0x01,0x31b,0x1a9a },
	{ 77,0,1,0,0x1f81,0x4e16,0x00001052,0x001881,0x02,0x31b,0x1fe4 },
	{ 78,0,1,0,0x0492,0x300c,0x000622e7,0x0a2f80,0x03,0x31b,0x0589 },
	{ 79,0,1,0,0x09a3,0x3202,0x0001e727,0x043bc0,0x04,0x31b,0x0c4a },
	{ 80,0,1,0,0x0eb4,0x33f8,0x000024c7,0x01c260,0x05,0x31b,0x1b34 },
	{ 81,0,0,0,0x11cf,0x33f8,0x000024c7,0x000000,0x05,0x31b,0x1b34 },
	{ 82,0,0,0,0x14ea,0x33f8,0x000024c7,0x000000,0x05,0x31b,0x1b34 },
	{ 83,0,0,0,0x1805,0x33f8,0x000024c7,0x000000,0x05,0x31b,0x1b34 },
	{ 84,0,0,0,0x1b20,0x33f8,0x000024c7,0x000000,0x05,0x31b,0x1b34 },
	{ 85,0,1,0,0x2031,0x35ee,0x00000f06,0x0015c1,0x06,0x31b,0x2000 },
	{ 86,0,1,0,0x0542,0x17e4,0x0005fb89,0x090b00,0x07,0x31b,0x05ae },
	{ 87,0,1,0,0x0a53,0x19da,0x000239c9,0x03c1c0,0x08,0x31b,0x0b63 },
	{ 88,0,1,0,0x0f64,0x1bd0,0x0000a9f9,0x018fd0,0x09,0x31b,0x125e },
	{ 89,0,1,0,0x1475,0x1dc6,0x000003c1,0x00a638,0x0a,0x31b,0x2000 },
	{ 90,0,0,0,0x1790,0x1dc6,0x000003c1,0x000000,0x0a,0x31b,0x2000 },
	{ 91,0,0,0,0x1aab,0x1dc6,0x000003c1,0x000000,0x0a,0x31b,0x2000 },
	{ 92,0,0,0,0x1dc6,0x58c7,0x000003c1,0x000000,0x00,0x31b,0x2000 },
	{ 93,0,0,0,0x00e1,0x38c7,0x0003c1ab,0x000000,0x00,0x31b,0x085e },
	{ 94,0,0,0,0x03fc,0x38c7,0x0003c1ab,0x000000,0x00,0x31b,0x085e },
	{ 95,0,0,0,0x0717,0x38c7,0x0003c1ab,0x000000,0x00,0x31b,0x085e },
	{ 96,0,1,0,0x0c28,0x3abd,0x0001054b,0x02bc60,0x01,0x31b,0x0fe3 },
	{ 97,0,0,0,0x0f43,0x3abd,0x0001054b,0x000000,0x01,0x31b,0x0fe3 },
	{ 98,0,1,0,0x1454,0x3cb3,0x00005b53,0x00a9f8,0x02,0x31b,0x15f4 },
	{ 99,0,1,0,0x1965,0x3ea9,0x000014af,0x0046a4,0x03,0x31b,0x1e86 },
	{ 100,0,0,0,0x1c80,0x3ea9,0x000014af,0x000000,0x03,0x31b,0x1e86 },
	{ 101,0,1,0,0x2191,0x409f,0x0000038a,0x001125,0x04,0x31b,0x2000 },
	{ 102,0,0,0,0x04ac,0x209f,0x00038acd,0x000000,0x04,0x31b,0x08b5 },
	{ 103,0,0,0,0x07c7,0x209f,0x00038acd,0x000000,0x04,0x31b,0x08b5 },
	{ 104,0,1,0,0x0cd8,0x2295,0x00011d0d,0x026dc0,0x05,0x31b,0x0f63 },
	{ 105,0,1,0,0x11e9,0x248b,0x00001aad,0x010260,0x06,0x31b,0x1d0e },
	{ 106,0,0,0,0x1504,0x248b,0x00001aad,0x000000,0x06,0x31b,0x1d0e },
	{ 107,0,0,0,0x181f,0x248b,0x00001aad,0x000000,0x06,0x31b,0x1d0e },
	{ 108,0,0,0,0x1b3a,0x248b,0x00001aad,0x000000,0x06,0x31b,0x1d0e },
	{ 109,0,1,0,0x204b,0x2681,0x0000054d,0x001560,0x07,0x31b,0x2000 },
	{ 110,0,0,0,0x0366,0x0681,0x00054def,0x000000,0x07,0x31b,0x0660 },
	{ 111,0,1,0,0x0877,0x0877,0x00001def,0x053000,0x08,0x31b,0x1c63 },
	{ 112,0,0,0,0x0b92,0x4693,0x00001def,0x000000,0x00,0x31b,0x1c63 },
	{ 113,0,0,0,0x0ead,0x4693,0x00001def,0x000000,0x00,0x31b,0x1c63 },
	{ 114,0,0,0,0x11c8,0x4693,0x00001def,0x000000,0x00,0x31b,0x1c63 },
	{ 115,0,0,0,0x14e3,0x4693,0x00001def,0x000000,0x00,0x31b,0x1c63 },
	{ 116,0,0,0,0x17fe,0x4693,0x00001def,0x000000,0x00,0x31b,0x1c63 },
	{ 117,0,0,0,0x1b19,0x4693,0x00001def,0x000000,0x00,0x31b,0x1c63 },
	{ 118,0,1,0,0x202a,0x4889,0x00000813,0x0015dc,0x01,0x31b,0x2000 },
	{ 119,0,0,0,0x0345,0x2889,0x00081301,0x000000,0x01,0x31b,0x03f3 },
	{ 120,0,1,0,0x0856,0x2a7f,0x0002c541,0x054dc0,0x02,0x31b,0x0a20 },
	{ 121,0,1,0,0x0d67,0x2c75,0x000090e1,0x023460,0x03,0x31b,0x134a },
	{ 122,0,0,0,0x1082,0x2c75,0x000090e1,0x000000,0x03,0x31b,0x134a },
	{ 123,0,1,0,0x1593,0x2e6b,0x000007e9,0x0088f8,0x04,0x31b,0x2000 },
	{ 124,0,0,0,0x18ae,0x2e6b,0x000007e9,0x000000,0x04,0x31b,0x2000 },
	{ 125,0,0,0,0x1bc9,0x2e6b,0x000007e9,0x000000,0x04,0x31b,0x2000 },
	{ 126,0,0,0,0x1ee4,0x2e6b,0x000007e9,0x000000,0x04,0x31b,0x2000 },
	{ 127,0,0,0,0x01ff,0x0e6b,0x0007e923,0x000000,0x04,0x31b,0x0411 },
	{ 128,0,1,0,0x0710,0x1061,0x00014c63,0x069cc0,0x05,0x31b,0x0e80 },
	{ 129,0,0,0,0x0a2b,0x1061,0x00014c63,0x000000,0x05,0x31b,0x0e80 },
	{ 130,0,0,0,0x0d46,0x1061,0x00014c63,0x000000,0x05,0x31b,0x0e80 },
	{ 131,0,1,0,0x1257,0x1257,0x00005c93,0x00efd0,0x06,0x31b,0x15e0 },
	{ 132,0,0,0,0x1572,0x5073,0x00005c93,0x000000,0x00,0x31b,0x15e0 },
	{ 133,0,1,0,0x1a83,0x5269,0x00002261,0x003a32,0x01,0x31b,0x1b98 },
	{ 134,0,1,0,0x1f94,0x545f,0x00000a30,0x001831,0x02,0x31b,0x2000 },
	{ 135,0,1,0,0x04a5,0x3655,0x000021c5,0x0a0e80,0x03,0x31b,0x1bb2 },
	{ 136,0,0,0,0x07c0,0x3655,0x000021c5,0x000000,0x03,0x31b,0x1bb2 },
	{ 137,0,0,0,0x0adb,0x3655,0x000021c5,0x000000,0x03,0x31b,0x1bb2 },
	{ 138,0,0,0,0x0df6,0x3655,0x000021c5,0x000000,0x03,0x31b,0x1bb2 },
	{ 139,0,0,0,0x1111,0x3655,0x000021c5,0x000000,0x03,0x31b,0x1bb2 },
	{ 140,0,0,0,0x142c,0x3655,0x000021c5,0x000000,0x03,0x31b,0x1bb2 },
	{ 141,0,0,0,0x1747,0x3655,0x000021c5,0x000000,0x03,0x31b,0x1bb2 },
	{ 142,0,0,0,0x1a62,0x3655,0x000021c5,0x000000,0x03,0x31b,0x1bb2 },
	{ 143,0,1,0,0x1f73,0x384b,0x00000908,0x0018bd,0x04,0x31b,0x2000 },
	{ 144,0,0,0,0x028e,0x184b,0x00090867,0x000000,0x04,0x31b,0x034e },
	{ 145,0,1,0,0x079f,0x1a41,0x000307a7,0x0600c0,0x05,0x31b,0x099c },
	{ 146,0,1,0,0x0cb0,0x1c37,0x000088c7,0x027ee0,0x06,0x31b,0x13a0 },
	{ 147,0,0,0,0x0fcb,0x1c37,0x000088c7,0x000000,0x06,0x31b,0x13a0 },
	{ 148,0,0,0,0x12e6,0x1c37,0x000088c7,0x000000,0x06,0x31b,0x13a0 },
	{ 149,0,1,0,0x17f7,0x1e2d,0x00002e47,0x005a80,0x07,0x31b,0x19e0 },
	{ 150,0,1,0,0x1d08,0x2023,0x000008a9,0x00259e,0x08,0x31b,0x2000 },
	{ 151,0,0,0,0x0023,0x3b24,0x0008a989,0x000000,0x00,0x31b,0x038c },
	{ 152,0,0,0,0x033e,0x3b24,0x0008a989,0x000000,0x00,0x31b,0x038c },
	{ 153,0,1,0,0x084f,0x3d1a,0x00035549,0x055440,0x01,0x31b,0x090e },
	{ 154,0,1,0,0x0d60,0x3f10,0x00011e29,0x023720,0x02,0x31b,0x0f5d },
	{ 155,0,1,0,0x1271,0x4106,0x00003289,0x00eba0,0x03,0x31b,0x195e },
	{ 156,0,0,0,0x158c,0x4106,0x00003289,0x000000,0x03,0x31b,0x195e },
	{ 157,0,0,0,0x18a7,0x4106,0x00003289,0x000000,0x03,0x31b,0x195e },
	{ 158,0,1,0,0x1db8,0x42fc,0x00001125,0x002164,0x04,0x31b,0x1f9b },
	{ 159,0,1,0,0x02c9,0x24f2,0x000345ab,0x0de000,0x05,0x31b,0x092a },
	{ 160,0,0,0,0x05e4,0x24f2,0x000345ab,0x000000,0x05,0x31b,0x092a },
	{ 161,0,0,0,0x08ff,0x24f2,0x000345ab,0x000000,0x05,0x31b,0x092a },
	{ 162,0,1,0,0x0e10,0x26e8,0x00014e7b,0x01f730,0x06,0x31b,0x0e76 },
	{ 163,0,1,0,0x1321,0x28de,0x00007d4b,0x00d130,0x07,0x31b,0x1421 },
	{ 164,0,1,0,0x1832,0x2ad4,0x00002657,0x0056f4,0x08,0x31b,0x1af6 },
	{ 165,0,1,0,0x1d43,0x2cca,0x00000231,0x002426,0x09,0x31b,0x2000 },
	{ 166,0,0,0,0x005e,0x0cca,0x000231cd,0x000000,0x09,0x31b,0x0b78 },
	{ 167,0,0,0,0x0379,0x0cca,0x000231cd,0x000000,0x09,0x31b,0x0b78 },
	{ 168,0,0,0,0x0694,0x0cca,0x000231cd,0x000000,0x09,0x31b,0x0b78 },
	{ 169,0,0,0,0x09af,0x0cca,0x000231cd,0x000000,0x09,0x31b,0x0b78 },
	{ 170,0,1,0,0x0ec0,0x0ec0,0x0000730d,0x01bec0,0x0a,0x31b,0x149e },
	{ 171,0,0,0,0x11db,0x4cdc,0x0000730d,0x000000,0x00,0x31b,0x149e },
	{ 172,0,1,0,0x16ec,0x4ed2,0x000006a5,0x006c68,0x01,0x31b,0x2000 },
	{ 173,0,0,0,0x1a07,0x4ed2,0x000006a5,0x000000,0x01,0x31b,0x2000 },
	{ 174,0,0,0,0x1d22,0x4ed2,0x000006a5,0x000000,0x01,0x31b,0x2000 },
	{ 175,0,0,0,0x003d,0x2ed2,0x0006a5ef,0x000000,0x01,0x31b,0x0513 },
	{ 176,0,0,0,0x0358,0x2ed2,0x0006a5ef,0x000000,0x01,0x31b,0x0513 },
	{ 177,0,1,0,0x0869,0x30c8,0x0001696f,0x053c80,0x02,0x31b,0x0e04 },
	{ 178,0,0,0,0x0b84,0x30c8,0x0001696f,0x000000,0x02,0x31b,0x0e04 },
	{ 179,0,1,0,0x1095,0x32be,0x0000241f,0x014550,0x03,0x31b,0x1b4f },
	{ 180,0,0,0,0x13b0,0x32be,0x0000241f,0x000000,0x03,0x31b,0x1b4f },
	{ 181,0,0,0,0x16cb,0x32be,0x0000241f,0x000000,0x03,0x31b,0x1b4f },
	{ 182,0,0,0,0x19e6,0x32be,0x0000241f,0x000000,0x03,0x31b,0x1b4f },
	{ 183,0,1,0,0x1ef7,0x34b4,0x00000938,0x001ae7,0x04,0x31b,0x2000 },
	{ 184,0,0,0,0x0212,0x14b4,0x00093801,0x000000,0x04,0x31b,0x0330 },
	{ 185,0,1,0,0x0723,0x16aa,0x0002b0c1,0x068740,0x05,0x31b,0x0a4b },
	{ 186,0,0,0,0x0a3e,0x16aa,0x0002b0c1,0x000000,0x05,0x31b,0x0a4b },
	{ 187,0,1,0,0x0f4f,0x18a0,0x00011b31,0x019590,0x06,0x31b,0x0f6c },
	{ 188,0,1,0,0x1460,0x1a96,0x00007299,0x00a898,0x07,0x31b,0x14a4 },
	{ 189,0,1,0,0x1971,0x1c8c,0x00002c85,0x004614,0x08,0x31b,0x1a1a },
	{ 190,0,1,0,0x1e82,0x1e82,0x00000f67,0x001d1e,0x09,0x31b,0x2000 },
	{ 191,0,1,0,0x0393,0x0393,0x00034ca3,0x0c1a80,0x0a,0x31b,0x091d },
	{ 192,0,0,0,0x06ae,0x41af,0x00034ca3,0x000000,0x00,0x31b,0x091d },
	{ 193,0,1,0,0x0bbf,0x43a5,0x00005cc3,0x02efe0,0x01,0x31b,0x15dd },
	{ 194,0,0,0,0x0eda,0x43a5,0x00005cc3,0x000000,0x01,0x31b,0x15dd },
	{ 195,0,0,0,0x11f5,0x43a5,0x00005cc3,0x000000,0x01,0x31b,0x15dd },
	{ 196,0,0,0,0x1510,0x43a5,0x00005cc3,0x000000,0x01,0x31b,0x15dd },
	{ 197,0,1,0,0x1a21,0x459b,0x00001e95,0x003e2e,0x02,0x31b,0x1c44 },
	{ 198,0,1,0,0x1f32,0x4791,0x000004bc,0x0019d9,0x03,0x31b,0x2000 },
	{ 199,0,0,0,0x024d,0x2791,0x0004bc45,0x000000,0x03,0x31b,0x0708 },
	{ 200,0,0,0,0x0568,0x2791,0x0004bc45,0x000000,0x03,0x31b,0x0708 },
	{ 201,0,1,0,0x0a79,0x2987,0x000112e5,0x03a960,0x04,0x31b,0x0f98 },
	{ 202,0,0,0,0x0d94,0x2987,0x000112e5,0x000000,0x04,0x31b,0x0f98 },
	{ 203,0,1,0,0x12a5,0x2b7d,0x00002f6d,0x00e378,0x05,0x31b,0x19bc },
	{ 204,0,0,0,0x15c0,0x2b7d,0x00002f6d,0x000000,0x05,0x31b,0x19bc },
	{ 205,0,0,0,0x18db,0x2b7d,0x00002f6d,0x000000,0x05,0x31b,0x19bc },
	{ 206,0,1,0,0x1dec,0x2d73,0x00000f33,0x00203a,0x06,0x31b,0x2000 },
	{ 207,0,1,0,0x02fd,0x0f69,0x0001cde7,0x0d6580,0x07,0x31b,0x0c99 },
	{ 208,0,0,0,0x0618,0x0f69,0x0001cde7,0x000000,0x07,0x31b,0x0c99 },
	{ 209,0,0,0,0x0933,0x0f69,0x0001cde7,0x000000,0x07,0x31b,0x0c99 },
	{ 210,0,0,0,0x0c4e,0x0f69,0x0001cde7,0x000000,0x07,0x31b,0x0c99 },
	{ 211,0,1,0,0x115f,0x115f,0x0000b227,0x011bc0,0x08,0x31b,0x1219 },
	{ 212,0,1,0,0x1670,0x1670,0x00003c43,0x0075e4,0x09,0x31b,0x185a },
	{ 213,0,1,0,0x1b81,0x1b81,0x00000b41,0x003102,0x0a,0x31b,0x2000 },
	{ 214,0,0,0,0x1e9c,0x599d,0x00000b41,0x000000,0x00,0x31b,0x2000 },
	{ 215,0,0,0,0x01b7,0x399d,0x000b4189,0x000000,0x00,0x31b,0x0209 },
	{ 216,0,1,0,0x06c8,0x3b93,0x00045049,0x06f140,0x01,0x31b,0x0792 },
	{ 217,0,1,0,0x0bd9,0x3d89,0x00016d69,0x02e2e0,0x02,0x31b,0x0df4 },
	{ 218,0,1,0,0x10ea,0x3f7f,0x00003a49,0x013320,0x03,0x31b,0x188b },
	{ 219,0,0,0,0x1405,0x3f7f,0x00003a49,0x000000,0x03,0x31b,0x188b },
	{ 220,0,0,0,0x1720,0x3f7f,0x00003a49,0x000000,0x03,0x31b,0x188b },
	{ 221,0,1,0,0x1c31,0x4175,0x00000ec7,0x002b82,0x04,0x31b,0x2000 },
	{ 222,0,0,0,0x1f4c,0x4175,0x00000ec7,0x000000,0x04,0x31b,0x2000 },
	{ 223,0,1,0,0x045d,0x236b,0x000438ab,0x0a8f00,0x05,0x31b,0x07b2 },
	{ 224,0,0,0,0x0778,0x236b,0x000438ab,0x000000,0x05,0x31b,0x07b2 },
	{ 225,0,1,0,0x0c89,0x2561,0x0001a8cb,0x028fe0,0x06,0x31b,0x0d15 },
	{ 226,0,1,0,0x119a,0x2757,0x0000982b,0x0110a0,0x07,0x31b,0x1302 },
	{ 227,0,1,0,0x16ab,0x294d,0x000026e3,0x007148,0x08,0x31b,0x1ae2 },
	{ 228,0,0,0,0x19c6,0x294d,0x000026e3,0x000000,0x08,0x31b,0x1ae2 },
	{ 229,0,1,0,0x1ed7,0x2b43,0x00000b66,0x001b7d,0x09,0x31b,0x2000 },
	{ 230,0,0,0,0x01f2,0x0b43,0x000b66cd,0x000000,0x09,0x31b,0x01f6 },
	{ 231,0,1,0,0x0703,0x0d39,0x0004bb0d,0x06abc0,0x0a,0x31b,0x070a },
	{ 232,0,1,0,0x0c14,0x4b02,0x0001f52d,0x02c5e0,0x00,0x37f,0x0c21 },
	{ 233,0,1,0,0x10a3,0x4c12,0x0000e15d,0x0113d0,0x01,0x37f,0x10bd },
	{ 234,0,1,0,0x1532,0x4d22,0x0000643d,0x007d20,0x02,0x37f,0x156a },
	{ 235,0,1,0,0x19c1,0x4e32,0x00002b73,0x0038ca,0x03,0x37f,0x1a3d },
	{ 236,0,1,0,0x1e50,0x4f42,0x000011ac,0x0019c7,0x04,0x37f,0x1f6f },
	{ 237,0,1,0,0x02df,0x3052,0x0005f96f,0x0bb380,0x05,0x37f,0x05b1 },
	{ 238,0,1,0,0x076e,0x3162,0x0000a9af,0x054fc0,0x06,0x37f,0x1261 },
	{ 239,0,0,0,0x0aed,0x3162,0x0000a9af,0x000000,0x06,0x37f,0x1261 },
	{ 240,0,0,0,0x0e6c,0x3162,0x0000a9af,0x000000,0x06,0x37f,0x1261 },
	{ 241,0,0,0,0x11eb,0x3162,0x0000a9af,0x000000,0x06,0x37f,0x1261 },
	{ 242,0,1,0,0x167a,0x3272,0x00004573,0x00643c,0x07,0x37f,0x1789 },
	{ 243,0,1,0,0x1b09,0x3382,0x000017f5,0x002d7e,0x08,0x37f,0x1dad },
	{ 244,0,1,0,0x1f98,0x3492,0x0000034f,0x0014a6,0x09,0x37f,0x2000 },
	{ 245,0,0,0,0x0317,0x1492,0x00034f01,0x000000,0x09,0x37f,0x0919 },
	{ 246,0,0,0,0x0696,0x1492,0x00034f01,0x000000,0x09,0x37f,0x0919 },
	{ 247,0,1,0,0x0b25,0x15a2,0x000084c1,0x02ca40,0x0a,0x37f,0x13cb },
	{ 248,0,0,0,0x0ea4,0x15a2,0x000084c1,0x000000,0x0a,0x37f,0x13cb },
	{ 249,0,0,0,0x1223,0x15a2,0x000084c1,0x000000,0x0a,0x37f,0x13cb },
	{ 250,0,1,1,0x16b2,0x56b2,0x00002441,0x006080,0x00,0x400,0x1b49 },
	{ 251,0,1,1,0x1ab2,0x56b2,0x00002441,0x000000,0x00,0x400,0x1b49 },
	{ 252,0,0,1,0x1eb2,0x56b2,0x0000102f,0x001412,0x01,0x400,0x1ff0 },
	{ 253,0,0,1,0x02b2,0x36b2,0x00062623,0x0a0900,0x02,0x400,0x0586 },
	{ 254,0,0,1,0x06b2,0x36b2,0x000121a3,0x050480,0x03,0x400,0x0f4b },
	{ 255,0,1,1,0x0ab2,0x36b2,0x000121a3,0x000000,0x03,0x400,0x0f4b },
	{ 256,0,1,1,0x0eb2,0x36b2,0x000121a3,0x000000,0x03,0x400,0x0f4b },
	{ 257,0,0,1,0x12b2,0x36b2,0x00008113,0x00a090,0x04,0x400,0x13f5 },
	{ 258,0,0,1,0x16b2,0x36b2,0x000030cb,0x005048,0x05,0x400,0x1992 },
	{ 259,0,0,1,0x1ab2,0x36b2,0x000008a7,0x002824,0x06,0x400,0x2000 },
	{ 260,0,1,1,0x1eb2,0x36b2,0x000008a7,0x000000,0x06,0x400,0x2000 },
	{ 261,0,1,1,0x02b2,0x16b2,0x0008a745,0x000000,0x06,0x400,0x038e },
	{ 262,0,0,1,0x06b2,0x16b2,0x0003a2c5,0x050480,0x07,0x400,0x088e },
	{ 263,0,0,1,0x0ab2,0x16b2,0x00012085,0x028240,0x08,0x400,0x0f51 },
	{ 264,0,1,1,0x0eb2,0x16b2,0x00012085,0x000000,0x08,0x400,0x0f51 },
	{ 265,0,0,1,0x12b2,0x16b2,0x00007ff5,0x00a090,0x09,0x400,0x1401 },
	{ 266,0,0,1,0x16b2,0x16b2,0x00002fad,0x005048,0x0a,0x400,0x19b4 },
	{ 267,0,0,0,0x1ab2,0x59a0,0x00000789,0x002824,0x00,0x37f,0x2000 },
	{ 268,0,0,0,0x1e31,0x59a0,0x00000789,0x000000,0x00,0x37f,0x2000 },
	{ 269,0,0,0,0x01b0,0x39a0,0x00078967,0x000000,0x00,0x37f,0x0459 },
	{ 270,0,1,0,0x063f,0x3ab0,0x00010467,0x068500,0x01,0x37f,0x0fe8 },
	{ 271,0,0,0,0x09be,0x3ab0,0x00010467,0x000000,0x01,0x37f,0x0fe8 },
	{ 272,0,0,0,0x0d3d,0x3ab0,0x00010467,0x000000,0x01,0x37f,0x0fe8 },
	{ 273,0,1,0,0x11cc,0x3bc0,0x000022ef,0x00e178,0x02,0x37f,0x1b80 },
	{ 274,0,0,0,0x154b,0x3bc0,0x000022ef,0x000000,0x02,0x37f,0x1b80 },
	{ 275,0,0,0,0x18ca,0x3bc0,0x000022ef,0x000000,0x02,0x37f,0x1b80 },
	{ 276,0,1,0,0x1d59,0x3cd0,0x00000478,0x001e77,0x03,0x37f,0x2000 },
	{ 277,0,0,0,0x00d8,0x1cd0,0x00047889,0x000000,0x03,0x37f,0x075d },
	{ 278,0,0,0,0x0457,0x1cd0,0x00047889,0x000000,0x03,0x37f,0x075d },
	{ 279,0,1,0,0x08e6,0x1de0,0x00005a89,0x041e00,0x04,0x37f,0x1601 },
	{ 280,0,0,0,0x0c65,0x1de0,0x00005a89,0x000000,0x04,0x37f,0x1601 },
	{ 281,0,0,0,0x0fe4,0x1de0,0x00005a89,0x000000,0x04,0x37f,0x1601 },
	{ 282,0,0,0,0x1363,0x1de0,0x00005a89,0x000000,0x04,0x37f,0x1601 },
	{ 283,0,1,0,0x17f2,0x1ef0,0x00000cd1,0x004db8,0x05,0x37f,0x2000 },
	{ 284,0,0,0,0x1b71,0x1ef0,0x00000cd1,0x000000,0x05,0x37f,0x2000 },
	{ 285,0,0,0,0x1ef0,0x59f1,0x00000cd1,0x000000,0x00,0x31b,0x2000 },
	{ 286,0,1,0,0x0401,0x3be7,0x0001952b,0x0b3c80,0x01,0x31b,0x0d5b },
	{ 287,0,0,0,0x071c,0x3be7,0x0001952b,0x000000,0x01,0x31b,0x0d5b },
	{ 288,0,0,0,0x0a37,0x3be7,0x0001952b,0x000000,0x01,0x31b,0x0d5b },
	{ 289,0,0,0,0x0d52,0x3be7,0x0001952b,0x000000,0x01,0x31b,0x0d5b },
	{ 290,0,1,0,0x1263,0x3ddd,0x0000a74b,0x00ede0,0x02,0x31b,0x1276 },
	{ 291,0,1,0,0x1774,0x3fd3,0x0000446b,0x0062e0,0x03,0x31b,0x179f },
	{ 292,0,1,0,0x1c85,0x41c9,0x00001b51,0x00291a,0x04,0x31b,0x1ceb },
	{ 293,0,1,0,0x2196,0x43bf,0x00000a3b,0x001116,0x05,0x31b,0x2000 },
	{ 294,0,1,0,0x06a7,0x25b5,0x0003228d,0x071940,0x06,0x31b,0x0969 },
};

uint8_t stream4[]={ 0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef, };

struct KnownState knownstates4[]=
{
//	 i,s,yn,mps,  lr,   lrm,         x,      dx,   k,   lp,    lx
	{ 0,-1,-1,-1,0x1001,0x1001,0x00000123,-1,-1,-1,-1 },
	{ 1,0,0,0,0x1401,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 2,1,0,0,0x1801,0x5401,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 3,2,0,0,0x1c01,0x5801,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 4,3,0,0,0x0001,0x3c01,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 5,0,0,0,0x0401,0x3c01,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 6,1,0,0,0x0801,0x4001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 7,2,0,0,0x0c01,0x4401,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 8,3,1,0,0x1001,0x4801,0x0000235d,0x00ffe8,0x01,0x400,0x1b6e },
	{ 9,0,0,0,0x1401,0x4801,0x0000235d,0x000000,0x00,0x400,0x1b6e },
	{ 10,1,0,0,0x1801,0x4c01,0x0000235d,0x000000,0x00,0x400,0x1b6e },
	{ 11,2,1,0,0x1c01,0x5001,0x00000360,0x001ffd,0x01,0x400,0x2000 },
	{ 12,3,0,0,0x0001,0x3401,0x00036067,0x000000,0x01,0x400,0x08fb },
	{ 13,0,0,0,0x0401,0x3401,0x00036067,0x000000,0x00,0x400,0x08fb },
	{ 14,1,0,0,0x0801,0x3801,0x00036067,0x000000,0x00,0x400,0x08fb },
	{ 15,2,1,0,0x0c01,0x3c01,0x00016097,0x01ffd0,0x02,0x400,0x0e28 },
	{ 16,3,1,0,0x1001,0x4001,0x000060af,0x00ffe8,0x02,0x400,0x15a0 },
	{ 17,0,0,0,0x1401,0x4001,0x000060af,0x000000,0x00,0x400,0x15a0 },
	{ 18,1,1,0,0x1801,0x4401,0x000020b5,0x003ffa,0x01,0x400,0x1be1 },
	{ 19,2,1,0,0x1c01,0x4801,0x000000b8,0x001ffd,0x03,0x400,0x2000 },
	{ 20,3,0,0,0x0001,0x2c01,0x0000b889,0x000000,0x02,0x400,0x11e5 },
	{ 21,0,0,0,0x0401,0x2c01,0x0000b889,0x000000,0x00,0x400,0x11e5 },
	{ 22,1,0,0,0x0801,0x3001,0x0000b889,0x000000,0x01,0x400,0x11e5 },
	{ 23,2,0,0,0x0c01,0x3401,0x0000b889,0x000000,0x03,0x400,0x11e5 },
	{ 24,3,0,0,0x1001,0x3801,0x0000b889,0x000000,0x02,0x400,0x11e5 },
	{ 25,0,1,0,0x1401,0x3801,0x00003895,0x007ff4,0x01,0x400,0x18b7 },
	{ 26,1,0,0,0x1801,0x3c01,0x00003895,0x000000,0x01,0x400,0x18b7 },
	{ 27,2,1,0,0x1c01,0x4001,0x00001898,0x001ffd,0x04,0x400,0x1d86 },
	{ 28,3,1,0,0x0001,0x2401,0x00089a2b,0x0ffe80,0x03,0x400,0x0396 },
	{ 29,0,1,0,0x0401,0x2401,0x00009aeb,0x07ff40,0x02,0x400,0x12e7 },
	{ 30,1,0,0,0x0801,0x2801,0x00009aeb,0x000000,0x01,0x400,0x12e7 },
	{ 31,2,0,0,0x0c01,0x2c01,0x00009aeb,0x000000,0x04,0x400,0x12e7 },
	{ 32,3,0,0,0x1001,0x3001,0x00009aeb,0x000000,0x03,0x400,0x12e7 },
	{ 33,0,1,0,0x1401,0x3001,0x00001af7,0x007ff4,0x03,0x400,0x1cfe },
	{ 34,1,0,0,0x1801,0x3401,0x00001af7,0x000000,0x01,0x400,0x1cfe },
	{ 35,2,0,0,0x1c01,0x3801,0x00001af7,0x000000,0x04,0x400,0x1cfe },
	{ 36,3,1,0,0x0001,0x1c01,0x000af94d,0x0ffe80,0x04,0x400,0x022e },
	{ 37,0,1,0,0x0401,0x1c01,0x0002fa0d,0x07ff40,0x04,0x400,0x09b6 },
	{ 38,1,0,0,0x0801,0x2001,0x0002fa0d,0x000000,0x01,0x400,0x09b6 },
	{ 39,2,1,0,0x0c01,0x2401,0x0000fa3d,0x01ffd0,0x05,0x400,0x1023 },
	{ 40,3,0,0,0x1001,0x2801,0x0000fa3d,0x000000,0x04,0x400,0x1023 },
	{ 41,0,1,0,0x1401,0x2801,0x00007a49,0x007ff4,0x05,0x400,0x1444 },
	{ 42,1,1,0,0x1801,0x2c01,0x00003a4f,0x003ffa,0x02,0x400,0x188b },
	{ 43,2,1,0,0x1c01,0x3001,0x00001a52,0x001ffd,0x06,0x400,0x1d22 },
	{ 44,3,1,0,0x0001,0x1401,0x000a546f,0x0ffe80,0x05,0x400,0x0288 },
	{ 45,0,1,0,0x0401,0x1401,0x0002552f,0x07ff40,0x06,0x400,0x0b1e },
	{ 46,1,0,0,0x0801,0x1801,0x0002552f,0x000000,0x02,0x400,0x0b1e },
	{ 47,2,1,0,0x0c01,0x1c01,0x0000555f,0x01ffd0,0x07,0x400,0x1658 },
	{ 48,3,0,0,0x1001,0x2001,0x0000555f,0x000000,0x05,0x400,0x1658 },
	{ 49,0,0,0,0x1401,0x2001,0x0000555f,0x000000,0x06,0x400,0x1658 },
	{ 50,1,1,0,0x1801,0x2401,0x00001565,0x003ffa,0x03,0x400,0x1e54 },
	{ 51,2,0,0,0x1c01,0x2801,0x00001565,0x000000,0x07,0x400,0x1e54 },
	{ 52,3,1,0,0x0001,0x0c01,0x00056681,0x0ffe80,0x06,0x400,0x0646 },
	{ 53,0,0,0,0x0401,0x0c01,0x00056681,0x000000,0x06,0x400,0x0646 },
	{ 54,1,1,0,0x0801,0x1001,0x000166e1,0x03ffa0,0x04,0x400,0x0e0e },
	{ 55,2,0,0,0x0c01,0x1401,0x000166e1,0x000000,0x07,0x400,0x0e0e },
	{ 56,3,1,0,0x1001,0x1801,0x000066f9,0x00ffe8,0x07,0x400,0x1542 },
	{ 57,0,0,0,0x1401,0x1801,0x000066f9,0x000000,0x06,0x400,0x1542 },
	{ 58,1,1,0,0x1801,0x1c01,0x000026ff,0x003ffa,0x05,0x400,0x1add },
	{ 59,2,1,0,0x1c01,0x2001,0x00000702,0x001ffd,0x08,0x400,0x2000 },
	{ 60,3,0,0,0x0001,0x0401,0x00070223,0x000000,0x07,0x400,0x04c5 },
	{ 61,0,0,0,0x0401,0x4401,0x00070223,0x000000,0x00,0x400,0x04c5 },
	{ 62,1,1,0,0x0801,0x0801,0x00030283,0x03ffa0,0x06,0x400,0x09a5 },
	{ 63,2,1,0,0x0c01,0x0c01,0x000102b3,0x01ffd0,0x09,0x400,0x0ff1 },
	{ 64,3,1,0,0x1001,0x1001,0x000002cb,0x00ffe8,0x08,0x400,0x2000 },
	{ 65,0,0,0,0x1401,0x5001,0x000002cb,0x000000,0x00,0x400,0x2000 },
	{ 66,1,0,0,0x1801,0x5801,0x000002cb,0x000000,0x00,0x400,0x2000 },
	{ 67,2,0,0,0x1c01,0x5c01,0x000002cb,0x000000,0x00,0x400,0x2000 },
	{ 68,3,0,0,0x0001,0x4001,0x0002cb45,0x000000,0x00,0x400,0x0a13 },
	{ 69,0,0,0,0x0401,0x3c01,0x0002cb45,0x000000,0x00,0x400,0x0a13 },
	{ 70,1,0,0,0x0801,0x4401,0x0002cb45,0x000000,0x00,0x400,0x0a13 },
	{ 71,2,1,0,0x0c01,0x4801,0x0000cb75,0x01ffd0,0x01,0x400,0x1155 },
	{ 72,3,0,0,0x1001,0x4c01,0x0000cb75,0x000000,0x00,0x400,0x1155 },
	{ 73,0,1,0,0x1401,0x4801,0x00004b81,0x007ff4,0x01,0x400,0x170d },
	{ 74,1,1,0,0x1801,0x5001,0x00000b87,0x003ffa,0x01,0x400,0x2000 },
	{ 75,2,0,0,0x1c01,0x5401,0x00000b87,0x000000,0x01,0x400,0x2000 },
	{ 76,3,0,0,0x0001,0x3801,0x000b8767,0x000000,0x00,0x400,0x01e6 },
	{ 77,0,1,0,0x0401,0x3401,0x00038827,0x07ff40,0x02,0x400,0x08b9 },
	{ 78,1,0,0,0x0801,0x3c01,0x00038827,0x000000,0x01,0x400,0x08b9 },
	{ 79,2,1,0,0x0c01,0x4001,0x00018857,0x01ffd0,0x02,0x400,0x0d8a },
	{ 80,3,1,0,0x1001,0x4401,0x0000886f,0x00ffe8,0x01,0x400,0x13a3 },
	{ 81,0,1,0,0x1401,0x4001,0x0000087b,0x007ff4,0x03,0x400,0x2000 },
	{ 82,1,0,0,0x1801,0x4801,0x0000087b,0x000000,0x01,0x400,0x2000 },
	{ 83,2,0,0,0x1c01,0x4c01,0x0000087b,0x000000,0x02,0x400,0x2000 },
	{ 84,3,0,0,0x0001,0x3001,0x00087b89,0x000000,0x01,0x400,0x03ab },
	{ 85,0,1,0,0x0401,0x2c01,0x00007c49,0x07ff40,0x04,0x400,0x142c },
	{ 86,1,0,0,0x0801,0x3401,0x00007c49,0x000000,0x01,0x400,0x142c },
	{ 87,2,0,0,0x0c01,0x3801,0x00007c49,0x000000,0x02,0x400,0x142c },
	{ 88,3,0,0,0x1001,0x3c01,0x00007c49,0x000000,0x01,0x400,0x142c },
	{ 89,0,0,0,0x1401,0x3801,0x00007c49,0x000000,0x04,0x400,0x142c },
	{ 90,1,1,0,0x1801,0x4001,0x00003c4f,0x003ffa,0x02,0x400,0x1859 },
	{ 91,2,1,0,0x1c01,0x4401,0x00001c52,0x001ffd,0x03,0x400,0x1cb5 },
	{ 92,3,1,0,0x0001,0x2801,0x000c542b,0x0ffe80,0x02,0x400,0x0182 },
	{ 93,0,1,0,0x0401,0x2401,0x000454eb,0x07ff40,0x05,0x400,0x078c },
	{ 94,1,1,0,0x0801,0x2c01,0x0000554b,0x03ffa0,0x03,0x400,0x1659 },
	{ 95,2,0,0,0x0c01,0x3001,0x0000554b,0x000000,0x03,0x400,0x1659 },
	{ 96,3,0,0,0x1001,0x3401,0x0000554b,0x000000,0x02,0x400,0x1659 },
	{ 97,0,0,0,0x1401,0x3001,0x0000554b,0x000000,0x05,0x400,0x1659 },
	{ 98,1,1,0,0x1801,0x3801,0x00001551,0x003ffa,0x04,0x400,0x1e59 },
	{ 99,2,0,0,0x1c01,0x3c01,0x00001551,0x000000,0x03,0x400,0x1e59 },
	{ 100,3,1,0,0x0001,0x2001,0x0005534d,0x0ffe80,0x03,0x400,0x065b },
	{ 101,0,0,0,0x0401,0x1c01,0x0005534d,0x000000,0x05,0x400,0x065b },
	{ 102,1,1,0,0x0801,0x2401,0x000153ad,0x03ffa0,0x05,0x400,0x0e60 },
	{ 103,2,0,0,0x0c01,0x2801,0x000153ad,0x000000,0x03,0x400,0x0e60 },
	{ 104,3,1,0,0x1001,0x2c01,0x000053c5,0x00ffe8,0x04,0x400,0x1674 },
	{ 105,0,0,0,0x1401,0x2801,0x000053c5,0x000000,0x05,0x400,0x1674 },
	{ 106,1,1,0,0x1801,0x3001,0x000013cb,0x003ffa,0x06,0x400,0x1ec7 },
	{ 107,2,0,0,0x1c01,0x3401,0x000013cb,0x000000,0x03,0x400,0x1ec7 },
	{ 108,3,1,0,0x0001,0x1801,0x0003cd6f,0x0ffe80,0x05,0x400,0x084c },
	{ 109,0,0,0,0x0401,0x1401,0x0003cd6f,0x000000,0x05,0x400,0x084c },
	{ 110,1,0,0,0x0801,0x1c01,0x0003cd6f,0x000000,0x06,0x400,0x084c },
	{ 111,2,1,0,0x0c01,0x2001,0x0001cd9f,0x01ffd0,0x04,0x400,0x0c9a },
	{ 112,3,1,0,0x1001,0x2401,0x0000cdb7,0x00ffe8,0x06,0x400,0x1144 },
	{ 113,0,1,0,0x1401,0x2001,0x00004dc3,0x007ff4,0x06,0x400,0x16e2 },
	{ 114,1,1,0,0x1801,0x2801,0x00000dc9,0x003ffa,0x07,0x400,0x2000 },
	{ 115,2,0,0,0x1c01,0x2c01,0x00000dc9,0x000000,0x04,0x400,0x2000 },
	{ 116,3,0,0,0x0001,0x1001,0x000dc901,0x000000,0x06,0x400,0x00dd },
	{ 117,0,1,0,0x0401,0x0c01,0x0005c9c1,0x07ff40,0x07,0x400,0x05df },
	{ 118,1,1,0,0x0801,0x1401,0x0001ca21,0x03ffa0,0x08,0x400,0x0ca5 },
	{ 119,2,0,0,0x0c01,0x1801,0x0001ca21,0x000000,0x04,0x400,0x0ca5 },
	{ 120,3,1,0,0x1001,0x1c01,0x0000ca39,0x00ffe8,0x07,0x400,0x115d },
	{ 121,0,1,0,0x1401,0x1801,0x00004a45,0x007ff4,0x08,0x400,0x1726 },
	{ 122,1,1,0,0x1801,0x2001,0x00000a4b,0x003ffa,0x09,0x400,0x2000 },
	{ 123,2,0,0,0x1c01,0x2401,0x00000a4b,0x000000,0x04,0x400,0x2000 },
	{ 124,3,0,0,0x0001,0x0801,0x000a4b23,0x000000,0x07,0x400,0x028d },
	{ 125,0,1,0,0x0401,0x0401,0x00024be3,0x07ff40,0x09,0x400,0x0b35 },
	{ 126,1,0,0,0x0801,0x0c01,0x00024be3,0x000000,0x09,0x400,0x0b35 },
	{ 127,2,1,0,0x0c01,0x1001,0x00004c13,0x01ffd0,0x05,0x400,0x1702 },
	{ 128,3,0,0,0x1001,0x1401,0x00004c13,0x000000,0x07,0x400,0x1702 },
	{ 129,0,0,0,0x1401,0x5401,0x00004c13,0x000000,0x00,0x400,0x1702 },
	{ 130,1,1,0,0x1801,0x1801,0x00000c19,0x003ffa,0x0a,0x400,0x2000 },
	{ 131,2,0,0,0x1c01,0x5aef,0x00000c19,0x000000,0x00,0x37f,0x2000 },
	{ 132,3,0,0,0x0001,0x4001,0x000c1945,0x000000,0x00,0x400,0x019e },
	{ 133,0,1,0,0x0401,0x4001,0x00041a05,0x07ff40,0x01,0x400,0x07dc },
	{ 134,1,1,1,0x0801,0x46ef,0x00001a65,0x03ffa0,0x00,0x37f,0x1d1d },
	{ 135,2,0,0,0x0b80,0x46ef,0x00001a65,0x000000,0x00,0x37f,0x1d1d },
	{ 136,3,0,0,0x0f80,0x4b80,0x00001a65,0x000000,0x00,0x400,0x1d1d },
	{ 137,0,0,0,0x1380,0x4b80,0x00001a65,0x000000,0x01,0x400,0x1d1d },
	{ 138,1,1,1,0x16ff,0x526e,0x00001a65,0x000000,0x00,0x37f,0x1d1d },
	{ 139,2,0,0,0x1a7e,0x526e,0x00001a65,0x000000,0x00,0x37f,0x1d1d },
	{ 140,3,1,0,0x1e7e,0x567e,0x0000059b,0x0014ca,0x01,0x400,0x2000 },
	{ 141,0,0,0,0x027e,0x367e,0x00059b67,0x000000,0x01,0x400,0x060e },
	{ 142,1,1,1,0x05fd,0x3ded,0x00059b67,0x000000,0x00,0x37f,0x060e },
	{ 143,2,1,0,0x0a8c,0x3efd,0x00028347,0x031820,0x01,0x37f,0x0ab0 },
	{ 144,3,1,0,0x0e8c,0x428c,0x000139c7,0x014980,0x02,0x400,0x0ed5 },
	{ 145,0,1,0,0x128c,0x428c,0x00009507,0x00a4c0,0x02,0x400,0x1321 },
	{ 146,1,0,1,0x171b,0x4b8c,0x00003b23,0x0059e4,0x01,0x37f,0x1876 },
	{ 147,2,1,0,0x1baa,0x4c9c,0x00001257,0x0028cc,0x02,0x37f,0x1f38 },
	{ 148,3,1,0,0x1faa,0x4faa,0x0000015f,0x0010f8,0x03,0x400,0x2000 },
	{ 149,0,0,0,0x03aa,0x2faa,0x00015f89,0x000000,0x02,0x400,0x0e2d },
	{ 150,1,1,1,0x0729,0x381b,0x00015f89,0x000000,0x01,0x37f,0x0e2d },
	{ 151,2,0,0,0x0aa8,0x381b,0x00015f89,0x000000,0x02,0x37f,0x0e2d },
	{ 152,3,1,0,0x0ea8,0x3aa8,0x00001c39,0x014350,0x04,0x400,0x1cba },
	{ 153,0,0,0,0x12a8,0x3aa8,0x00001c39,0x000000,0x02,0x400,0x1cba },
	{ 154,1,1,1,0x1627,0x439a,0x00001c39,0x000000,0x01,0x37f,0x1cba },
	{ 155,2,0,0,0x19a6,0x439a,0x00001c39,0x000000,0x02,0x37f,0x1cba },
	{ 156,3,1,0,0x1da6,0x45a6,0x0000042a,0x00180f,0x05,0x400,0x2000 },
	{ 157,0,0,0,0x01a6,0x25a6,0x00042aab,0x000000,0x02,0x400,0x07c5 },
	{ 158,1,1,1,0x0525,0x2f19,0x00042aab,0x000000,0x01,0x37f,0x07c5 },
	{ 159,2,1,0,0x09b4,0x3029,0x000095eb,0x0394c0,0x03,0x37f,0x1318 },
	{ 160,3,0,0,0x0db4,0x31b4,0x000095eb,0x000000,0x05,0x400,0x1318 },
	{ 161,0,0,0,0x11b4,0x31b4,0x000095eb,0x000000,0x02,0x400,0x1318 },
	{ 162,1,0,1,0x1643,0x3cb8,0x00002de3,0x006808,0x02,0x37f,0x19ed },
	{ 163,2,0,0,0x19c2,0x3cb8,0x00002de3,0x000000,0x03,0x37f,0x19ed },
	{ 164,3,1,0,0x1dc2,0x3dc2,0x00001647,0x00179c,0x06,0x400,0x1e18 },
	{ 165,0,1,0,0x01c2,0x1dc2,0x000a79cd,0x0bce00,0x03,0x400,0x0273 },
	{ 166,1,0,1,0x0651,0x2947,0x0004090d,0x0670c0,0x03,0x37f,0x07f4 },
	{ 167,2,1,0,0x0ae0,0x2a57,0x00011cad,0x02ec60,0x04,0x37f,0x0f65 },
	{ 168,3,0,0,0x0ee0,0x2ae0,0x00011cad,0x000000,0x06,0x400,0x0f65 },
	{ 169,0,1,0,0x12e0,0x2ae0,0x00008105,0x009ba8,0x04,0x400,0x13f5 },
	{ 170,1,0,1,0x176f,0x36e6,0x00002c19,0x0054ec,0x04,0x37f,0x1a28 },
	{ 171,2,1,0,0x1bfe,0x37f6,0x0000058d,0x00268c,0x05,0x37f,0x2000 },
	{ 172,3,0,0,0x1ffe,0x37fe,0x0000058d,0x000000,0x06,0x400,0x2000 },
	{ 173,0,0,0,0x03fe,0x17fe,0x00058def,0x000000,0x04,0x400,0x061c },
	{ 174,1,0,1,0x088d,0x2485,0x00012e2f,0x045fc0,0x05,0x37f,0x0f0d },
	{ 175,2,0,0,0x0c0c,0x2485,0x00012e2f,0x000000,0x05,0x37f,0x0f0d },
	{ 176,3,1,0,0x100c,0x240c,0x0000302f,0x00fe00,0x07,0x400,0x19a5 },
	{ 177,0,0,0,0x140c,0x240c,0x0000302f,0x000000,0x04,0x400,0x19a5 },
	{ 178,1,1,1,0x178b,0x3004,0x0000302f,0x000000,0x05,0x37f,0x19a5 },
	{ 179,2,1,0,0x1c1a,0x3114,0x00000a5d,0x0025d2,0x06,0x37f,0x2000 },
	{ 180,3,0,0,0x001a,0x101a,0x000a5d01,0x000000,0x07,0x400,0x0283 },
	{ 181,0,1,0,0x041a,0x101a,0x00028041,0x07dcc0,0x05,0x400,0x0ab7 },
	{ 182,1,1,1,0x0799,0x1c93,0x00028041,0x000000,0x05,0x37f,0x0ab7 },
	{ 183,2,1,0,0x0c28,0x1da3,0x000028c1,0x025780,0x07,0x37f,0x1a9c },
	{ 184,3,0,0,0x1028,0x1c28,0x000028c1,0x000000,0x07,0x400,0x1a9c },
	{ 185,0,0,0,0x1428,0x1c28,0x000028c1,0x000000,0x05,0x400,0x1a9c },
	{ 186,1,1,1,0x17a7,0x2922,0x000028c1,0x000000,0x05,0x37f,0x1a9c },
	{ 187,2,1,0,0x1c36,0x2a32,0x000003a5,0x00251c,0x08,0x37f,0x2000 },
	{ 188,3,0,0,0x0036,0x0836,0x0003a523,0x000000,0x07,0x400,0x088a },
	{ 189,0,0,0,0x0436,0x0836,0x0003a523,0x000000,0x05,0x400,0x088a },
	{ 190,1,1,1,0x07b5,0x15b1,0x0003a523,0x000000,0x05,0x37f,0x088a },
	{ 191,2,1,0,0x0c44,0x16c1,0x000158e3,0x024c40,0x09,0x37f,0x0e49 },
	{ 192,3,1,0,0x1044,0x1444,0x00006453,0x00f490,0x08,0x400,0x1569 },
	{ 193,0,0,0,0x1444,0x5332,0x00006453,0x000000,0x00,0x37f,0x1569 },
	{ 194,1,0,1,0x18d3,0x2350,0x00002197,0x0042bc,0x06,0x37f,0x1bba },
	{ 195,2,1,0,0x1d62,0x2460,0x0000034f,0x001e48,0x0a,0x37f,0x2000 },
	{ 196,3,0,0,0x0162,0x4162,0x00034f45,0x000000,0x00,0x400,0x0919 },
	{ 197,0,0,0,0x04e1,0x4050,0x00034f45,0x000000,0x00,0x37f,0x0919 },
	{ 198,1,1,1,0x0860,0x0f5e,0x00034f45,0x000000,0x06,0x37f,0x0919 },
	{ 199,2,1,1,0x0cef,0x4cef,0x00014385,0x020bc0,0x00,0x400,0x0ea8 },
	{ 200,3,1,0,0x10ef,0x4cef,0x000069ad,0x00d9d8,0x01,0x400,0x151c },
	{ 201,0,0,0,0x146e,0x4c5e,0x000069ad,0x000000,0x00,0x37f,0x151c },
	{ 202,1,0,1,0x18fd,0x1c7c,0x000028d1,0x0040dc,0x07,0x37f,0x1a9a },
	{ 203,2,0,1,0x1cfd,0x58fd,0x00000dd8,0x001af9,0x01,0x400,0x2000 },
	{ 204,3,1,0,0x00fd,0x38fd,0x00005be7,0x0d7c80,0x02,0x400,0x15eb },
	{ 205,0,0,0,0x047c,0x38ed,0x00005be7,0x000000,0x00,0x37f,0x15eb },
	{ 206,1,1,1,0x07fb,0x46e9,0x00005be7,0x000000,0x00,0x37f,0x15eb },
	{ 207,2,1,1,0x0bfb,0x43fb,0x00005be7,0x000000,0x01,0x400,0x15eb },
	{ 208,3,0,0,0x0ffb,0x43fb,0x00005be7,0x000000,0x02,0x400,0x15eb },
	{ 209,0,0,0,0x137a,0x446c,0x00005be7,0x000000,0x00,0x37f,0x15eb },
	{ 210,1,0,1,0x1809,0x5378,0x00000f63,0x004c84,0x01,0x37f,0x2000 },
	{ 211,2,1,1,0x1c09,0x5009,0x00000f63,0x000000,0x01,0x400,0x2000 },
	{ 212,3,0,0,0x0009,0x3009,0x000f6389,0x000000,0x02,0x400,0x003a },
	{ 213,0,1,0,0x0498,0x320b,0x0006b489,0x08af00,0x01,0x37f,0x0506 },
	{ 214,1,0,1,0x0927,0x4117,0x0002c429,0x03f060,0x02,0x37f,0x0a22 },
	{ 215,2,0,1,0x0d27,0x3d27,0x000120a9,0x01a380,0x02,0x400,0x0f50 },
	{ 216,3,1,0,0x1127,0x3d27,0x00004ee9,0x00d1c0,0x03,0x400,0x16cc },
	{ 217,0,0,0,0x14a6,0x3e9a,0x00004ee9,0x000000,0x01,0x37f,0x16cc },
	{ 218,1,0,1,0x1935,0x4da6,0x0000107b,0x003e6e,0x03,0x37f,0x1fd5 },
	{ 219,2,1,1,0x1d35,0x4935,0x0000107b,0x000000,0x02,0x400,0x1fd5 },
	{ 220,3,1,0,0x0135,0x2935,0x00037f2b,0x0cfc80,0x04,0x400,0x08c8 },
	{ 221,0,0,0,0x04b4,0x2b29,0x00037f2b,0x000000,0x01,0x37f,0x08c8 },
	{ 222,1,1,1,0x0833,0x3925,0x00037f2b,0x000000,0x03,0x37f,0x08c8 },
	{ 223,2,0,1,0x0c33,0x3433,0x0001906b,0x01eec0,0x03,0x400,0x0d6c },
	{ 224,3,1,0,0x1033,0x3433,0x0000990b,0x00f760,0x05,0x400,0x12f9 },
	{ 225,0,1,0,0x14c2,0x37b8,0x0000120b,0x008700,0x02,0x37f,0x1f50 },
	{ 226,1,1,1,0x1841,0x45b4,0x0000120b,0x000000,0x03,0x37f,0x1f50 },
	{ 227,2,1,1,0x1c41,0x4041,0x0000120b,0x000000,0x03,0x400,0x1f50 },
	{ 228,3,1,0,0x0041,0x2041,0x0002bb4d,0x0f5080,0x06,0x400,0x0a35 },
	{ 229,0,0,0,0x03c0,0x2337,0x0002bb4d,0x000000,0x02,0x37f,0x0a35 },
	{ 230,1,1,1,0x073f,0x3133,0x0002bb4d,0x000000,0x03,0x37f,0x0a35 },
	{ 231,2,0,1,0x0b3f,0x2b3f,0x0000736d,0x0247e0,0x04,0x400,0x149a },
	{ 232,3,0,0,0x0f3f,0x2b3f,0x0000736d,0x000000,0x06,0x400,0x149a },
	{ 233,0,0,0,0x12be,0x2eb6,0x0000736d,0x000000,0x02,0x37f,0x149a },
	{ 234,1,0,1,0x174d,0x3dc2,0x00001c89,0x0056e4,0x04,0x37f,0x1caa },
	{ 235,2,1,1,0x1b4d,0x374d,0x00001c89,0x000000,0x04,0x400,0x1caa },
	{ 236,3,1,0,0x1f4d,0x374d,0x00000a76,0x001213,0x07,0x400,0x2000 },
	{ 237,0,1,0,0x03dc,0x1c55,0x00009a6f,0x09dc80,0x03,0x37f,0x12ec },
	{ 238,1,1,1,0x075b,0x2a51,0x00009a6f,0x000000,0x04,0x37f,0x12ec },
	{ 239,2,1,1,0x0b5b,0x235b,0x00009a6f,0x000000,0x04,0x400,0x12ec },
	{ 240,3,0,0,0x0f5b,0x235b,0x00009a6f,0x000000,0x07,0x400,0x12ec },
	{ 241,0,0,0,0x12da,0x27d4,0x00009a6f,0x000000,0x03,0x37f,0x12ec },
	{ 242,1,0,1,0x1769,0x36e0,0x0000452b,0x005544,0x05,0x37f,0x178f },
	{ 243,2,0,1,0x1b69,0x2f69,0x000021b3,0x002378,0x05,0x400,0x1bb5 },
	{ 244,3,1,0,0x1f69,0x2f69,0x00000ff7,0x0011bc,0x08,0x400,0x2000 },
	{ 245,0,1,0,0x03f8,0x1573,0x00064a01,0x09ad00,0x04,0x37f,0x0565 },
	{ 246,1,0,1,0x0887,0x247f,0x0001e5c1,0x046440,0x06,0x37f,0x0c4f },
	{ 247,2,0,1,0x0c87,0x1c87,0x00001251,0x01d370,0x06,0x400,0x1f3a },
	{ 248,3,0,0,0x1087,0x1c87,0x00001251,0x000000,0x08,0x400,0x1f3a },
	{ 249,0,0,0,0x1406,0x2202,0x00001251,0x000000,0x04,0x37f,0x1f3a },
	{ 250,1,1,1,0x1785,0x2ffe,0x00001251,0x000000,0x06,0x37f,0x1f3a },
	{ 251,2,1,1,0x1b85,0x2785,0x00001251,0x000000,0x06,0x400,0x1f3a },
	{ 252,3,1,0,0x1f85,0x2785,0x000000ea,0x001167,0x09,0x400,0x2000 },
	{ 253,0,0,0,0x0304,0x0d81,0x0000ea23,0x000000,0x04,0x37f,0x1085 },
	{ 254,1,1,1,0x0683,0x1b7d,0x0000ea23,0x000000,0x06,0x37f,0x1085 },
	{ 255,2,1,1,0x0a83,0x1283,0x0000ea23,0x000000,0x06,0x400,0x1085 },
	{ 256,3,0,0,0x0e83,0x1283,0x0000ea23,0x000000,0x09,0x400,0x1085 },
	{ 257,0,1,0,0x1312,0x1a10,0x00003543,0x00b4e0,0x05,0x37f,0x1910 },
	{ 258,1,1,1,0x1691,0x280c,0x00003543,0x000000,0x06,0x37f,0x1910 },
	{ 259,2,0,1,0x1a91,0x1e91,0x00000c37,0x00290c,0x07,0x400,0x2000 },
	{ 260,3,0,0,0x1e91,0x5e91,0x00000c37,0x000000,0x00,0x400,0x2000 },
	{ 261,0,1,0,0x0320,0x069f,0x000104c5,0x0b3280,0x06,0x37f,0x0fe6 },
	{ 262,1,1,1,0x069f,0x149b,0x000104c5,0x000000,0x06,0x37f,0x0fe6 },
	{ 263,2,1,1,0x0a9f,0x4a9f,0x000104c5,0x000000,0x00,0x400,0x0fe6 },
	{ 264,3,0,0,0x0e9f,0x4a9f,0x000104c5,0x000000,0x00,0x400,0x0fe6 },
	{ 265,0,1,0,0x132e,0x132e,0x0000534d,0x00b178,0x07,0x37f,0x167c },
	{ 266,1,0,1,0x17bd,0x223a,0x000002bd,0x005090,0x07,0x37f,0x2000 },
	{ 267,2,1,1,0x1bbd,0x57bd,0x000002bd,0x000000,0x00,0x400,0x2000 },
	{ 268,3,0,0,0x1fbd,0x57bd,0x000002bd,0x000000,0x00,0x400,0x2000 },
	{ 269,0,0,0,0x033c,0x422a,0x0002bd67,0x000000,0x00,0x37f,0x0a30 },
	{ 270,1,1,1,0x06bb,0x0db9,0x0002bd67,0x000000,0x07,0x37f,0x0a30 },
	{ 271,2,0,1,0x0abb,0x42bb,0x00003f07,0x027e60,0x01,0x400,0x1817 },
	{ 272,3,0,0,0x0ebb,0x42bb,0x00003f07,0x000000,0x00,0x400,0x1817 },
	{ 273,0,0,0,0x123a,0x4da9,0x00003f07,0x000000,0x00,0x37f,0x1817 },
	{ 274,1,1,1,0x15b9,0x1938,0x00003f07,0x000000,0x07,0x37f,0x1817 },
	{ 275,2,0,1,0x19b9,0x4db9,0x00000f85,0x002f82,0x02,0x400,0x2000 },
	{ 276,3,0,0,0x1db9,0x4db9,0x00000f85,0x000000,0x00,0x400,0x2000 },
	{ 277,0,1,0,0x0248,0x3a38,0x00029009,0x0cf580,0x01,0x37f,0x0a93 },
	{ 278,1,1,1,0x05c7,0x44b5,0x00029009,0x000000,0x00,0x37f,0x0a93 },
	{ 279,2,1,1,0x09c7,0x39c7,0x00029009,0x000000,0x02,0x400,0x0a93 },
	{ 280,3,1,0,0x0dc7,0x39c7,0x00011789,0x017880,0x01,0x400,0x0f80 },
	{ 281,0,1,0,0x1256,0x46c7,0x00004a21,0x00cd68,0x02,0x37f,0x1728 },
	{ 282,1,1,1,0x15d5,0x5144,0x00004a21,0x000000,0x00,0x37f,0x1728 },
	{ 283,2,0,1,0x19d5,0x45d5,0x00001b83,0x002e9e,0x03,0x400,0x1ce0 },
	{ 284,3,1,0,0x1dd5,0x45d5,0x00000434,0x00174f,0x02,0x400,0x2000 },
	{ 285,0,0,0,0x0154,0x3246,0x000434ab,0x000000,0x02,0x37f,0x07b7 },
	{ 286,1,1,1,0x04d3,0x3cc3,0x000434ab,0x000000,0x00,0x37f,0x07b7 },
	{ 287,2,0,1,0x08d3,0x30d3,0x0000bcab,0x037800,0x04,0x400,0x11c4 },
	{ 288,3,0,0,0x0cd3,0x30d3,0x0000bcab,0x000000,0x02,0x400,0x11c4 },
	{ 289,0,0,0,0x1052,0x3dc5,0x0000bcab,0x000000,0x02,0x37f,0x11c4 },
	{ 290,1,0,1,0x14e1,0x4952,0x0000387b,0x008430,0x01,0x37f,0x18ba },
	{ 291,2,0,1,0x18e1,0x3ce1,0x00000181,0x0036fa,0x05,0x400,0x2000 },
	{ 292,3,0,0,0x1ce1,0x3ce1,0x00000181,0x000000,0x02,0x400,0x2000 },
	{ 293,0,0,0,0x0060,0x2a54,0x000181cd,0x000000,0x02,0x37f,0x0da3 },
};

uint8_t stream5[]={  0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef, };

struct KnownState knownstates5[]=
{
//	 i,s,yn,mps,  lr,   lrm,         x,      dx,   k,   lp,    lx
	{ 0,-1,-1,-1,0x1001,0x1001,0x00000123,-1,-1,-1,-1 },
	{ 1,0,0,0,0x1401,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 2,0,0,0,0x1801,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 3,0,0,0,0x1c01,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 4,0,0,0,0x0001,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 5,1,0,0,0x0401,0x4001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 6,0,0,0,0x0801,0x3401,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 7,0,0,0,0x0c01,0x3401,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 8,0,1,0,0x1001,0x3401,0x0000235d,0x00ffe8,0x01,0x400,0x1b6e },
	{ 9,0,0,0,0x1401,0x3401,0x0000235d,0x000000,0x01,0x400,0x1b6e },
	{ 10,2,0,0,0x1801,0x5401,0x0000235d,0x000000,0x00,0x400,0x1b6e },
	{ 11,0,1,0,0x1c01,0x3801,0x00000360,0x001ffd,0x02,0x400,0x2000 },
	{ 12,0,0,0,0x0001,0x1801,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 13,0,0,0,0x0401,0x1801,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 14,0,0,0,0x0801,0x1801,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 15,1,1,0,0x0c01,0x4401,0x00016097,0x01ffd0,0x01,0x400,0x0e28 },
	{ 16,0,1,0,0x1001,0x1c01,0x000060af,0x00ffe8,0x03,0x400,0x15a0 },
	{ 17,0,0,0,0x1401,0x1c01,0x000060af,0x000000,0x03,0x400,0x15a0 },
	{ 18,0,1,0,0x1801,0x1c01,0x000020b5,0x003ffa,0x04,0x400,0x1be1 },
	{ 19,0,1,0,0x1c01,0x1c01,0x000000b8,0x001ffd,0x05,0x400,0x2000 },
	{ 20,2,0,0,0x0001,0x3801,0x0000b889,0x000000,0x00,0x400,0x11e5 },
	{ 21,0,0,0,0x0401,0x42ef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 22,0,0,0,0x0780,0x42ef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 23,0,0,0,0x0aff,0x42ef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 24,0,0,0,0x0e7e,0x42ef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 25,1,1,0,0x127e,0x467e,0x00001239,0x00a650,0x02,0x400,0x1f41 },
	{ 26,0,0,0,0x15fd,0x46ef,0x00001239,0x000000,0x00,0x37f,0x1f41 },
	{ 27,0,0,0,0x197c,0x46ef,0x00001239,0x000000,0x00,0x37f,0x1f41 },
	{ 28,0,0,0,0x1cfb,0x46ef,0x00001239,0x000000,0x00,0x37f,0x1f41 },
	{ 29,0,1,0,0x018a,0x27ff,0x00037d2b,0x0ebc80,0x01,0x37f,0x08cb },
	{ 30,2,0,0,0x058a,0x398a,0x00037d2b,0x000000,0x00,0x400,0x08cb },
	{ 31,0,1,0,0x0a19,0x2d0f,0x0000250b,0x035820,0x02,0x37f,0x1b29 },
	{ 32,0,0,0,0x0d98,0x2d0f,0x0000250b,0x000000,0x02,0x37f,0x1b29 },
	{ 33,0,0,0,0x1117,0x2d0f,0x0000250b,0x000000,0x02,0x37f,0x1b29 },
	{ 34,0,0,0,0x1496,0x2d0f,0x0000250b,0x000000,0x02,0x37f,0x1b29 },
	{ 35,1,0,0,0x1896,0x4896,0x0000250b,0x000000,0x02,0x400,0x1b29 },
	{ 36,0,1,0,0x1d25,0x321f,0x0000057d,0x001f8e,0x03,0x37f,0x2000 },
	{ 37,0,0,0,0x00a4,0x121f,0x00057dcd,0x000000,0x03,0x37f,0x062d },
	{ 38,0,0,0,0x0423,0x121f,0x00057dcd,0x000000,0x03,0x37f,0x062d },
	{ 39,0,1,0,0x08b2,0x132f,0x000139cd,0x044400,0x04,0x37f,0x0ed5 },
	{ 40,2,0,0,0x0cb2,0x3cb2,0x000139cd,0x000000,0x00,0x400,0x0ed5 },
	{ 41,0,1,0,0x1141,0x183f,0x00004215,0x00f7b8,0x05,0x37f,0x17d2 },
	{ 42,0,0,0,0x14c0,0x183f,0x00004215,0x000000,0x05,0x37f,0x17d2 },
	{ 43,0,1,0,0x194f,0x194f,0x000004bd,0x003d58,0x06,0x37f,0x2000 },
	{ 44,0,0,0,0x1cce,0x5bbc,0x000004bd,0x000000,0x00,0x37f,0x2000 },
	{ 45,1,0,0,0x00ce,0x2cce,0x0004bdef,0x000000,0x02,0x400,0x0706 },
	{ 46,0,0,0,0x044d,0x3fbc,0x0004bdef,0x000000,0x00,0x37f,0x0706 },
	{ 47,0,1,0,0x08dc,0x40cc,0x000098af,0x042540,0x01,0x37f,0x12fd },
	{ 48,0,0,0,0x0c5b,0x40cc,0x000098af,0x000000,0x01,0x37f,0x12fd },
	{ 49,0,0,0,0x0fda,0x40cc,0x000098af,0x000000,0x01,0x37f,0x12fd },
	{ 50,2,1,0,0x13da,0x3fda,0x00001547,0x008368,0x01,0x400,0x1e5c },
	{ 51,0,0,0,0x1759,0x44cc,0x00001547,0x000000,0x01,0x37f,0x1e5c },
	{ 52,0,0,0,0x1ad8,0x44cc,0x00001547,0x000000,0x01,0x37f,0x1e5c },
	{ 53,0,0,0,0x1e57,0x44cc,0x00001547,0x000000,0x01,0x37f,0x1e5c },
	{ 54,0,1,0,0x02e6,0x25dc,0x0009a181,0x0ba580,0x02,0x37f,0x02ef },
	{ 55,1,1,0,0x06e6,0x2ee6,0x0004c941,0x04d840,0x03,0x400,0x06f8 },
	{ 56,0,1,0,0x0b75,0x2aec,0x000224a1,0x02a4a0,0x03,0x37f,0x0b9b },
	{ 57,0,1,0,0x1004,0x2bfc,0x0000f181,0x013320,0x04,0x37f,0x1057 },
	{ 58,0,1,0,0x1493,0x2d0c,0x00006619,0x008b68,0x05,0x37f,0x154f },
	{ 59,0,1,0,0x1922,0x2e1c,0x000026db,0x003f3e,0x06,0x37f,0x1ae3 },
	{ 60,2,1,0,0x1d22,0x4522,0x00000c8c,0x001a4f,0x02,0x400,0x2000 },
	{ 61,0,0,0,0x00a1,0x121c,0x000c8c23,0x000000,0x06,0x37f,0x0168 },
	{ 62,0,1,0,0x0530,0x132c,0x0004b763,0x07d4c0,0x07,0x37f,0x070e },
	{ 63,0,1,0,0x09bf,0x143c,0x00012983,0x038de0,0x08,0x37f,0x0f23 },
	{ 64,0,0,0,0x0d3e,0x143c,0x00012983,0x000000,0x08,0x37f,0x0f23 },
	{ 65,1,1,0,0x113e,0x353e,0x00005b03,0x00ce80,0x04,0x400,0x15f9 },
	{ 66,0,0,0,0x14bd,0x183c,0x00005b03,0x000000,0x08,0x37f,0x15f9 },
	{ 67,0,1,0,0x194c,0x194c,0x00001d8b,0x003d78,0x09,0x37f,0x1c77 },
	{ 68,0,1,0,0x1ddb,0x1ddb,0x000001a4,0x001be7,0x0a,0x37f,0x2000 },
	{ 69,0,0,0,0x015a,0x4048,0x0001a445,0x000000,0x00,0x37f,0x0d25 },
	{ 70,2,0,0,0x055a,0x295a,0x0001a445,0x000000,0x02,0x400,0x0d25 },
	{ 71,0,0,0,0x08d9,0x4448,0x0001a445,0x000000,0x00,0x37f,0x0d25 },
	{ 72,0,0,0,0x0c58,0x4448,0x0001a445,0x000000,0x00,0x37f,0x0d25 },
	{ 73,0,1,0,0x10e7,0x4558,0x00009cf5,0x010750,0x01,0x37f,0x12d4 },
	{ 74,0,1,0,0x1576,0x4668,0x00002575,0x007780,0x02,0x37f,0x1b19 },
	{ 75,1,0,0,0x1976,0x3976,0x00002575,0x000000,0x04,0x400,0x1b19 },
	{ 76,0,1,0,0x1e05,0x4b78,0x00000a56,0x001b1f,0x03,0x37f,0x2000 },
	{ 77,0,0,0,0x0184,0x2b78,0x000a5667,0x000000,0x03,0x37f,0x0287 },
	{ 78,0,1,0,0x0613,0x2c88,0x00039f27,0x06b740,0x04,0x37f,0x0894 },
	{ 79,0,1,0,0x0aa2,0x2d98,0x000092c7,0x030c60,0x05,0x37f,0x1337 },
	{ 80,2,0,0,0x0ea2,0x2ea2,0x000092c7,0x000000,0x02,0x400,0x1337 },
	{ 81,0,0,0,0x1221,0x3198,0x000092c7,0x000000,0x05,0x37f,0x1337 },
	{ 82,0,1,0,0x16b0,0x32a8,0x00003223,0x0060a4,0x06,0x37f,0x196a },
	{ 83,0,1,0,0x1b3f,0x33b8,0x00000647,0x002bdc,0x07,0x37f,0x2000 },
	{ 84,0,0,0,0x1ebe,0x33b8,0x00000647,0x000000,0x07,0x37f,0x2000 },
	{ 85,1,0,0,0x02be,0x1ebe,0x00064789,0x000000,0x04,0x400,0x0567 },
	{ 86,0,1,0,0x074d,0x18c8,0x0000d949,0x056e40,0x08,0x37f,0x10f3 },
	{ 87,0,0,0,0x0acc,0x18c8,0x0000d949,0x000000,0x08,0x37f,0x10f3 },
	{ 88,0,0,0,0x0e4b,0x18c8,0x0000d949,0x000000,0x08,0x37f,0x10f3 },
	{ 89,0,1,0,0x12da,0x19d8,0x00001d71,0x00bbd8,0x09,0x37f,0x1c7c },
	{ 90,2,0,0,0x16da,0x32da,0x00001d71,0x000000,0x02,0x400,0x1c7c },
	{ 91,0,0,0,0x1a59,0x1dd8,0x00001d71,0x000000,0x09,0x37f,0x1c7c },
	{ 92,0,1,0,0x1ee8,0x1ee8,0x0000062e,0x001743,0x0a,0x37f,0x2000 },
	{ 93,0,0,0,0x0267,0x4155,0x00062eab,0x000000,0x00,0x37f,0x057e },
	{ 94,0,1,0,0x06f6,0x4265,0x00006beb,0x05c2c0,0x01,0x37f,0x14fd },
	{ 95,1,0,0,0x0af6,0x22f6,0x00006beb,0x000000,0x04,0x400,0x14fd },
	{ 96,0,0,0,0x0e75,0x4665,0x00006beb,0x000000,0x01,0x37f,0x14fd },
	{ 97,0,0,0,0x11f4,0x4665,0x00006beb,0x000000,0x01,0x37f,0x14fd },
	{ 98,0,1,0,0x1683,0x4775,0x0000084b,0x0063a0,0x02,0x37f,0x2000 },
	{ 99,0,0,0,0x1a02,0x4775,0x0000084b,0x000000,0x02,0x37f,0x2000 },
	{ 100,2,0,0,0x1e02,0x3602,0x0000084b,0x000000,0x02,0x400,0x2000 },
	{ 101,0,0,0,0x0181,0x2b75,0x00084bcd,0x000000,0x02,0x37f,0x03cc },
	{ 102,0,1,0,0x0610,0x2c85,0x0001910d,0x06bac0,0x03,0x37f,0x0d6a },
	{ 103,0,0,0,0x098f,0x2c85,0x0001910d,0x000000,0x03,0x37f,0x0d6a },
	{ 104,0,0,0,0x0d0e,0x2c85,0x0001910d,0x000000,0x03,0x37f,0x0d6a },
	{ 105,1,1,0,0x110e,0x250e,0x0000bbbd,0x00d550,0x05,0x400,0x11cb },
	{ 106,0,1,0,0x159d,0x3195,0x00004759,0x007464,0x04,0x37f,0x1761 },
	{ 107,0,1,0,0x1a2c,0x32a5,0x00001285,0x0034d4,0x05,0x37f,0x1f29 },
	{ 108,0,0,0,0x1dab,0x32a5,0x00001285,0x000000,0x05,0x37f,0x1f29 },
	{ 109,0,1,0,0x023a,0x13b5,0x000570ef,0x0d1500,0x06,0x37f,0x063b },
	{ 110,2,0,0,0x063a,0x1a3a,0x000570ef,0x000000,0x02,0x400,0x063b },
	{ 111,0,1,0,0x0ac9,0x18c5,0x000278cf,0x02f820,0x07,0x37f,0x0ac9 },
	{ 112,0,1,0,0x0f58,0x19d5,0x00011fcf,0x015900,0x08,0x37f,0x0f55 },
	{ 113,0,1,0,0x13e7,0x1ae5,0x00008337,0x009c98,0x09,0x37f,0x13dd },
	{ 114,0,1,0,0x1876,0x1bf5,0x00003c23,0x004714,0x0a,0x37f,0x185d },
	{ 115,1,1,0,0x1c76,0x2c76,0x00001e96,0x001d8d,0x06,0x400,0x1c44 },
	{ 116,0,1,1,0x2105,0x6105,0x00000e76,0x001020,0x00,0x400,0x2000 },
	{ 117,0,0,1,0x0505,0x4105,0x0007c101,0x06b500,0x01,0x400,0x042f },
	{ 118,0,0,1,0x0905,0x4105,0x00046681,0x035a80,0x02,0x400,0x0774 },
	{ 119,0,0,1,0x0d05,0x4105,0x0002b941,0x01ad40,0x03,0x400,0x0a39 },
	{ 120,2,1,0,0x1105,0x2105,0x0001e2a1,0x00d6a0,0x03,0x400,0x0c58 },
	{ 121,0,0,1,0x1505,0x4505,0x00017751,0x006b50,0x04,0x400,0x0dcc },
	{ 122,0,0,1,0x1905,0x4505,0x000141a9,0x0035a8,0x05,0x400,0x0eb0 },
	{ 123,0,0,1,0x1d05,0x4505,0x000126d5,0x001ad4,0x06,0x400,0x0f31 },
	{ 124,0,0,1,0x0105,0x2505,0x01196b23,0x0d6a00,0x07,0x400,0xfd55 },
	{ 125,1,1,0,0x0505,0x1105,0x0112b623,0x06b500,0x07,0x400,0xff1a },
	{ 126,0,0,1,0x0905,0x2905,0x010f5ba3,0x035a80,0x08,0x400,0xfc1f },
	{ 127,0,0,1,0x0d05,0x2905,0x010dae63,0x01ad40,0x09,0x400,0xfc70 },
	{ 128,0,0,1,0x1105,0x2905,0x010cd7c3,0x00d6a0,0x0a,0x400,0xfc9a },
	{ 129,0,0,0,0x1505,0x5006,0x010c6c73,0x006b50,0x00,0x31b,0xfcb0 },
	{ 130,2,1,0,0x1905,0x2505,0x010c36cb,0x0035a8,0x04,0x400,0xfcbb },
	{ 131,0,1,0,0x1e16,0x55fc,0x010c1778,0x001f53,0x01,0x31b,0xfcc1 },
	{ 132,0,1,0,0x0327,0x37f2,0x0c0a72c5,0x0d0580,0x02,0x31b,0xfd1b },
	{ 133,0,1,0,0x0838,0x39e8,0x0c050905,0x0569c0,0x03,0x31b,0xfe6d },
	{ 134,0,1,0,0x0d49,0x3bde,0x0c02c905,0x024000,0x04,0x31b,0xff14 },
	{ 135,1,1,0,0x1149,0x1949,0x0c01fc0d,0x00ccf8,0x08,0x400,0xff55 },
	{ 136,0,1,0,0x165a,0x41d4,0x0c018465,0x0077a8,0x05,0x31b,0xff7c },
	{ 137,0,1,0,0x1b6b,0x43ca,0x0c0152a7,0x0031be,0x06,0x31b,0xff8c },
	{ 138,0,1,0,0x207c,0x45c0,0x0c013df9,0x0014ae,0x07,0x31b,0xff93 },
	{ 139,0,1,0,0x058d,0x27b6,0x013560e7,0x089880,0x08,0x31b,0xfe55 },
	{ 140,2,1,0,0x098d,0x118d,0x013251c7,0x030f20,0x05,0x400,0xff3a },
	{ 141,0,1,0,0x0e9e,0x2dac,0x013088a7,0x01c920,0x09,0x31b,0xffd1 },
	{ 142,0,1,0,0x13af,0x2fa2,0x012fca9f,0x00be08,0x0a,0x31b,0xfc0b },
	{ 143,0,1,1,0x18c0,0x58c0,0x012f7b9f,0x004f00,0x00,0x400,0xfc19 },
	{ 144,0,0,1,0x1cc0,0x58c0,0x012f5f83,0x001c1c,0x01,0x400,0xfc1e },
	{ 145,1,1,0,0x00c0,0x04c0,0x2f517589,0x0e0e00,0x09,0x400,0xff81 },
	{ 146,0,0,1,0x04c0,0x3cc0,0x2f4a6e89,0x070700,0x02,0x400,0xfd1c },
	{ 147,0,0,1,0x08c0,0x3cc0,0x2f46eb09,0x038380,0x03,0x400,0xfdee },
	{ 148,0,0,1,0x0cc0,0x3cc0,0x2f452949,0x01c1c0,0x04,0x400,0xfe64 },
	{ 149,0,0,1,0x10c0,0x3cc0,0x2f444869,0x00e0e0,0x05,0x400,0xfea3 },
	{ 150,2,1,0,0x14c0,0x18c0,0x2f43d7f9,0x007070,0x06,0x400,0xfec4 },
	{ 151,0,0,1,0x18c0,0x40c0,0x2f439fc1,0x003838,0x06,0x400,0xfed4 },
	{ 152,0,0,1,0x1cc0,0x40c0,0x2f4383a5,0x001c1c,0x07,0x400,0xfedc },
	{ 153,0,0,1,0x00c0,0x20c0,0x437597ab,0x0e0e00,0x08,0x400,0xfe47 },
	{ 154,0,0,1,0x04c0,0x20c0,0x436e90ab,0x070700,0x09,0x400,0xfc45 },
	{ 155,1,1,0,0x08c0,0x08c0,0x436b0d2b,0x038380,0x0a,0x400,0xfcf9 },
	{ 156,0,0,1,0x0cc0,0x24c0,0x43694b6b,0x01c1c0,0x0a,0x400,0xfd5d },
	{ 157,0,0,0,0x10c0,0x4bc1,0x43686a8b,0x00e0e0,0x00,0x31b,0xfd91 },
	{ 158,0,1,0,0x15d1,0x4db7,0x4367e73b,0x008350,0x01,0x31b,0xfdb0 },
	{ 159,0,1,0,0x1ae2,0x4fad,0x4367b0a9,0x003692,0x02,0x31b,0xfdbd },
	{ 160,2,1,0,0x1ee2,0x1ee2,0x43679d3b,0x00136e,0x07,0x400,0xfdc2 },
	{ 161,0,1,0,0x03f3,0x35a3,0x6791e3cd,0x0b5800,0x03,0x31b,0xff5d },
	{ 162,0,1,0,0x0904,0x3799,0x678d2ccd,0x04b700,0x04,0x31b,0xfc8a },
	{ 163,0,1,0,0x0e15,0x398f,0x678b374d,0x01f580,0x05,0x31b,0xfcf0 },
	{ 164,0,1,0,0x1326,0x3b85,0x678a66d5,0x00d078,0x06,0x31b,0xfd1d },
	{ 165,1,1,1,0x1726,0x5614,0x678a1c9d,0x004a38,0x00,0x37f,0xfd2e },
	{ 166,0,1,0,0x1c37,0x417b,0x6789f149,0x002b54,0x07,0x31b,0xfd37 },
	{ 167,0,1,0,0x2148,0x4371,0x6789df46,0x001203,0x08,0x31b,0xfd3b },
	{ 168,0,1,0,0x0659,0x2567,0x89d7cb2f,0x077bc0,0x09,0x31b,0xfdb7 },
	{ 169,0,1,0,0x0b6a,0x275d,0x89d4aecf,0x031c60,0x0a,0x31b,0xfe86 },
	{ 170,2,1,0,0x0f6a,0x0f6a,0x89d3933f,0x011b90,0x08,0x400,0xfed7 },
	{ 171,0,1,1,0x147b,0x547b,0x89d2edb7,0x00a588,0x00,0x400,0xff09 },
	{ 172,0,0,1,0x187b,0x547b,0x89d2b2cf,0x003ae8,0x01,0x400,0xff1b },
	{ 173,0,0,1,0x1c7b,0x547b,0x89d2955b,0x001d74,0x02,0x400,0xff24 },
	{ 174,0,0,1,0x007b,0x347b,0xd286a101,0x0eba00,0x03,0x400,0xfe01 },
	{ 175,1,0,1,0x050a,0x4079,0xd27e9801,0x080900,0x01,0x37f,0xfc43 },
	{ 176,0,0,1,0x090a,0x390a,0xd27b4061,0x0357a0,0x04,0x400,0xfcee },
	{ 177,0,0,1,0x0d0a,0x390a,0xd2799491,0x01abd0,0x05,0x400,0xfd4c },
	{ 178,0,0,1,0x110a,0x390a,0xd278bea9,0x00d5e8,0x06,0x400,0xfd7d },
	{ 179,0,0,1,0x150a,0x390a,0xd27853b5,0x006af4,0x07,0x400,0xfd96 },
	{ 180,2,1,0,0x190a,0x190a,0xd2781e3b,0x00357a,0x09,0x400,0xfda3 },
	{ 181,0,0,1,0x1d0a,0x3d0a,0xd278037e,0x001abd,0x08,0x400,0xfda9 },
	{ 182,0,0,1,0x010a,0x1d0a,0x77f61fa3,0x0d5e80,0x09,0x400,0xfe23 },
	{ 183,0,0,1,0x050a,0x1d0a,0x77ef7063,0x06af40,0x0a,0x400,0xfc1b },
	{ 184,0,0,0,0x090a,0x440b,0x77ec18c3,0x0357a0,0x00,0x31b,0xfcc1 },
	{ 185,1,0,1,0x0d99,0x4589,0x77ea45f3,0x01d2d0,0x02,0x37f,0xfd25 },
	{ 186,0,1,0,0x12aa,0x4a90,0x77e9633b,0x00e2b8,0x01,0x31b,0xfd57 },
	{ 187,0,1,0,0x17bb,0x4c86,0x77e904fb,0x005e40,0x02,0x31b,0xfd6d },
	{ 188,0,1,0,0x1ccc,0x4e7c,0x77e8ddcd,0x00272e,0x03,0x31b,0xfd76 },
	{ 189,0,1,0,0x21dd,0x5072,0x77e8cd85,0x001048,0x04,0x31b,0xfd7a },
	{ 190,2,1,0,0x05dd,0x05dd,0xe8c7b985,0x05cbc0,0x0a,0x400,0xfdbb },
	{ 191,0,1,0,0x0aee,0x3668,0xe8c45765,0x036220,0x05,0x31b,0xfe9f },
	{ 192,0,1,0,0x0fff,0x385e,0xe8c2ef55,0x016810,0x06,0x31b,0xff09 },
	{ 193,0,1,0,0x1510,0x3a54,0xe8c259ad,0x0095a8,0x07,0x31b,0xff37 },
	{ 194,0,1,0,0x1a21,0x3c4a,0xe8c21b7f,0x003e2e,0x08,0x31b,0xff4b },
	{ 195,1,0,1,0x1eb0,0x5321,0xe8c20356,0x001829,0x03,0x37f,0xff52 },
	{ 196,0,1,0,0x03c1,0x22cf,0xc1f79ae7,0x0bbb80,0x09,0x31b,0xfdc3 },
	{ 197,0,1,0,0x08d2,0x24c5,0xc1f2ba67,0x04e080,0x0a,0x31b,0xff19 },
	{ 198,0,1,1,0x0de3,0x4de3,0xc1f0b387,0x0206e0,0x00,0x400,0xffc2 },
	{ 199,0,0,1,0x11e3,0x4de3,0xc1effacf,0x00b8b8,0x01,0x400,0xfc02 },
	{ 200,2,1,1,0x15e3,0x54d1,0xc1ef9e73,0x005c5c,0x00,0x37f,0xfc13 },
	{ 201,0,0,1,0x19e3,0x51e3,0xc1ef7045,0x002e2e,0x02,0x400,0xfc1b },
	{ 202,0,0,1,0x1de3,0x51e3,0xc1ef592e,0x001717,0x03,0x400,0xfc1f },
	{ 203,0,0,1,0x01e3,0x31e3,0xef4da309,0x0b8b80,0x04,0x400,0xfc72 },
	{ 204,0,0,1,0x05e3,0x31e3,0xef47dd49,0x05c5c0,0x05,0x400,0xfdb3 },
	{ 205,1,0,1,0x0a72,0x3b64,0xef44b729,0x032620,0x04,0x37f,0xfe84 },
	{ 206,0,0,1,0x0e72,0x3672,0xef4367c9,0x014f60,0x06,0x400,0xfee4 },
	{ 207,0,0,1,0x1272,0x3672,0xef42c019,0x00a7b0,0x07,0x400,0xff17 },
	{ 208,0,0,1,0x1672,0x3672,0xef426c41,0x0053d8,0x08,0x400,0xff31 },
	{ 209,0,0,1,0x1a72,0x3672,0xef424255,0x0029ec,0x09,0x400,0xff3e },
	{ 210,2,0,1,0x1f01,0x5a70,0xef422b76,0x0016df,0x01,0x37f,0xff46 },
	{ 211,0,0,1,0x0301,0x1b01,0x4221f32b,0x098380,0x0a,0x400,0xff58 },
	{ 212,0,0,0,0x0701,0x4202,0x421d316b,0x04c1c0,0x00,0x31b,0xfc89 },
	{ 213,0,1,0,0x0c12,0x43f8,0x421a6a8b,0x02c6e0,0x01,0x31b,0xfd1c },
	{ 214,0,1,0,0x1123,0x45ee,0x4219430b,0x012780,0x02,0x31b,0xfd5e },
	{ 215,1,0,1,0x15b2,0x4325,0x4218d04b,0x0072c0,0x05,0x37f,0xfd79 },
	{ 216,0,1,0,0x1ac3,0x4c73,0x42189891,0x0037ba,0x03,0x31b,0xfd86 },
	{ 217,0,1,0,0x1fd4,0x4e69,0x42188166,0x00172b,0x04,0x31b,0xfd8b },
	{ 218,0,1,0,0x04e5,0x305f,0x1877c54d,0x09a180,0x05,0x31b,0xfdb8 },
	{ 219,0,1,0,0x09f6,0x3255,0x1873c54d,0x040000,0x06,0x31b,0xfec9 },
	{ 220,2,0,1,0x0e85,0x4675,0x1872375d,0x018df0,0x02,0x37f,0xff42 },
	{ 221,0,1,0,0x1396,0x38da,0x18717615,0x00c148,0x07,0x31b,0xff80 },
	{ 222,0,1,0,0x18a7,0x3ad0,0x187125bd,0x005058,0x08,0x31b,0xff9b },
	{ 223,0,1,0,0x1db8,0x3cc6,0x18710459,0x002164,0x09,0x31b,0xffa6 },
	{ 224,0,1,0,0x02c9,0x1ebc,0x70f679ef,0x0de000,0x0a,0x31b,0xfe0b },
	{ 225,1,0,1,0x0758,0x314c,0x70f115ef,0x056400,0x06,0x37f,0xffa1 },
	{ 226,0,1,1,0x0c69,0x4c69,0x70ee77af,0x029e40,0x00,0x400,0xfc49 },
	{ 227,0,0,1,0x1069,0x4c69,0x70ed892f,0x00ee80,0x01,0x400,0xfc77 },
	{ 228,0,0,1,0x1469,0x4c69,0x70ed11ef,0x007740,0x02,0x400,0xfc8f },
	{ 229,0,0,1,0x1869,0x4c69,0x70ecd64f,0x003ba0,0x03,0x400,0xfc9b },
	{ 230,2,0,1,0x1cf8,0x5169,0x70ecb5c5,0x00208a,0x03,0x37f,0xfca1 },
	{ 231,0,0,1,0x00f8,0x30f8,0xeca83c81,0x0d8880,0x04,0x400,0xfd9c },
	{ 232,0,0,1,0x04f8,0x30f8,0xeca17841,0x06c440,0x05,0x400,0xff80 },
	{ 233,0,0,1,0x08f8,0x30f8,0xec9e1621,0x036220,0x06,0x400,0xfc5c },
	{ 234,0,0,1,0x0cf8,0x30f8,0xec9c6511,0x01b110,0x07,0x400,0xfcb1 },
	{ 235,1,0,1,0x1187,0x37fc,0xec9b78d1,0x00ec40,0x07,0x37f,0xfce2 },
	{ 236,0,0,1,0x1587,0x3587,0xec9b1689,0x006248,0x08,0x400,0xfcf7 },
	{ 237,0,0,1,0x1987,0x3587,0xec9ae565,0x003124,0x09,0x400,0xfd02 },
	{ 238,0,0,1,0x1d87,0x3587,0xec9accd3,0x001892,0x0a,0x400,0xfd07 },
	{ 239,0,0,0,0x0187,0x3c88,0x9ac08a23,0x0c4900,0x00,0x31b,0xffd0 },
	{ 240,2,0,1,0x0616,0x3708,0x9ab9d663,0x06b3c0,0x04,0x37f,0xfd3d },
	{ 241,0,1,0,0x0b27,0x430d,0x9ab69503,0x034160,0x01,0x31b,0xfe04 },
	{ 242,0,1,0,0x1038,0x4503,0x9ab53a93,0x015a70,0x02,0x31b,0xfe60 },
	{ 243,0,1,0,0x1549,0x46f9,0x9ab4aa93,0x009000,0x03,0x31b,0xfe87 },
	{ 244,0,1,0,0x1a5a,0x48ef,0x9ab46ebf,0x003bd4,0x04,0x31b,0xfe98 },
	{ 245,1,0,1,0x1ee9,0x41df,0x9ab45780,0x00173f,0x08,0x37f,0xfe9f },
	{ 246,0,1,0,0x03fa,0x2f74,0xb44c3645,0x0b4a00,0x05,0x31b,0xfcbb },
	{ 247,0,1,0,0x090b,0x316a,0xb4478505,0x04b140,0x06,0x31b,0xfdc8 },
	{ 248,0,1,0,0x0e1c,0x3360,0xb44591e5,0x01f320,0x07,0x31b,0xfe48 },
	{ 249,0,1,0,0x132d,0x3556,0xb444c265,0x00cf80,0x08,0x31b,0xfe81 },
	{ 250,2,0,1,0x17bc,0x452f,0xb44471c9,0x00509c,0x05,0x37f,0xfe97 },
	{ 251,0,1,0,0x1ccd,0x3bdb,0xb4444aa3,0x002726,0x09,0x31b,0xfea2 },
	{ 252,0,1,0,0x21de,0x3dd1,0xb4443a5e,0x001045,0x0a,0x31b,0xfea7 },
	{ 253,0,1,1,0x06ef,0x46ef,0x44339b67,0x06c300,0x00,0x400,0xfed5 },
	{ 254,0,0,1,0x0aef,0x46ef,0x44313307,0x026860,0x01,0x400,0xff97 },
	{ 255,1,0,1,0x0f7e,0x2ef5,0x442fe2c7,0x015040,0x09,0x37f,0xfc06 },
	{ 256,0,0,1,0x137e,0x4b7e,0x442f56e7,0x008be0,0x02,0x400,0xfc20 },
	{ 257,0,0,1,0x177e,0x4b7e,0x442f10f7,0x0045f0,0x03,0x400,0xfc2d },
	{ 258,0,0,1,0x1b7e,0x4b7e,0x442eedff,0x0022f8,0x04,0x400,0xfc33 },
	{ 259,0,0,1,0x1f7e,0x4b7e,0x442edc83,0x00117c,0x05,0x400,0xfc36 },
	{ 260,2,0,1,0x040d,0x2e01,0x2ed2f989,0x098a00,0x06,0x37f,0xff06 },
	{ 261,0,0,1,0x080d,0x300d,0x2ecf0249,0x03f740,0x06,0x400,0xfc2f },
	{ 262,0,0,1,0x0c0d,0x300d,0x2ecd06a9,0x01fba0,0x07,0x400,0xfc91 },
	{ 263,0,0,1,0x100d,0x300d,0x2ecc08d9,0x00fdd0,0x08,0x400,0xfcc5 },
	{ 264,0,0,1,0x140d,0x300d,0x2ecb89f1,0x007ee8,0x09,0x400,0xfcdf },
	{ 265,1,0,1,0x189c,0x3494,0x2ecb44a9,0x004548,0x0a,0x37f,0xfced },
	{ 266,0,0,1,0x1c9c,0x349c,0x2ecb27dc,0x001ccd,0x0a,0x400,0xfcf4 },
	{ 267,0,0,0,0x009c,0x3b9d,0xcb19762b,0x0e6680,0x00,0x31b,0xfd53 },
	{ 268,0,1,0,0x05ad,0x3d93,0xcb110d2b,0x086900,0x01,0x31b,0xffa3 },
	{ 269,0,1,0,0x0abe,0x3f89,0xcb0d8e8b,0x037ea0,0x02,0x31b,0xfc76 },
	{ 270,2,0,1,0x0f4d,0x35c2,0xcb0c32fb,0x015b90,0x07,0x37f,0xfcbc },
	{ 271,0,1,0,0x145e,0x460e,0xcb0b8a2b,0x00a8d0,0x03,0x31b,0xfcdf },
	{ 272,0,1,0,0x196f,0x4804,0xcb0b43ff,0x00462c,0x04,0x31b,0xfcee },
	{ 273,0,1,0,0x1e80,0x49fa,0xcb0b26d7,0x001d28,0x05,0x31b,0xfcf4 },
	{ 274,0,1,0,0x0391,0x2bf0,0x0b1ab8cd,0x0c1f00,0x06,0x31b,0xfd0b },
};

uint8_t stream6[]={ 0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef, };

struct KnownState knownstates6[]=
{
//	 i,s,yn,mps,  lr,   lrm,         x,      dx,   k,   lp,    lx
	{ 0,-1,-1,-1,0x1001,0x1001,0x00000123,-1,-1,-1,-1 },
	{ 1,0,0,0,0x1401,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 2,0,0,0,0x1801,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 3,0,0,0,0x1c01,0x5001,0x00000123,0x000000,0x00,0x400,0x2000 },
	{ 4,0,0,0,0x0001,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 5,0,0,0,0x0401,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 6,0,0,0,0x0801,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 7,0,0,0,0x0c01,0x3001,0x00012345,0x000000,0x00,0x400,0x0f43 },
	{ 8,0,1,0,0x1001,0x3001,0x0000235d,0x00ffe8,0x01,0x400,0x1b6e },
	{ 9,0,0,0,0x1401,0x3001,0x0000235d,0x000000,0x01,0x400,0x1b6e },
	{ 10,0,0,0,0x1801,0x3001,0x0000235d,0x000000,0x01,0x400,0x1b6e },
	{ 11,0,1,0,0x1c01,0x3001,0x00000360,0x001ffd,0x02,0x400,0x2000 },
	{ 12,0,0,0,0x0001,0x1001,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 13,0,0,0,0x0401,0x1001,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 14,0,0,0,0x0801,0x1001,0x00036067,0x000000,0x02,0x400,0x08fb },
	{ 15,0,1,0,0x0c01,0x1001,0x00016097,0x01ffd0,0x03,0x400,0x0e28 },
	{ 16,1,1,0,0x1001,0x4c01,0x000060af,0x00ffe8,0x01,0x400,0x15a0 },
	{ 17,2,0,0,0x1401,0x5001,0x000060af,0x000000,0x00,0x400,0x15a0 },
	{ 18,3,1,0,0x1801,0x5401,0x000020b5,0x003ffa,0x01,0x400,0x1be1 },
	{ 19,0,1,0,0x1c01,0x1c01,0x000000b8,0x001ffd,0x04,0x400,0x2000 },
	{ 20,0,0,0,0x0001,0x3eef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 21,0,0,0,0x0380,0x3eef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 22,0,0,0,0x06ff,0x3eef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 23,0,0,0,0x0a7e,0x3eef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 24,0,0,0,0x0dfd,0x3eef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 25,0,0,0,0x117c,0x3eef,0x0000b889,0x000000,0x00,0x37f,0x11e5 },
	{ 26,0,1,0,0x160b,0x3fff,0x00004c7d,0x006c0c,0x01,0x37f,0x16fa },
	{ 27,0,1,0,0x1a9a,0x410f,0x00001b73,0x00310a,0x02,0x37f,0x1ce4 },
	{ 28,0,1,0,0x1f29,0x421f,0x00000530,0x001643,0x03,0x37f,0x2000 },
	{ 29,0,0,0,0x02a8,0x221f,0x000530ab,0x000000,0x03,0x37f,0x0681 },
	{ 30,0,0,0,0x0627,0x221f,0x000530ab,0x000000,0x03,0x37f,0x0681 },
	{ 31,0,1,0,0x0ab6,0x232f,0x00022ecb,0x0301e0,0x04,0x37f,0x0b80 },
	{ 32,0,1,0,0x0f45,0x243f,0x0000d14b,0x015d80,0x05,0x37f,0x112b },
	{ 33,0,1,0,0x13d4,0x254f,0x000032ab,0x009ea0,0x06,0x37f,0x195a },
	{ 34,1,0,0,0x17d4,0x4fd4,0x000032ab,0x000000,0x01,0x400,0x195a },
	{ 35,2,1,0,0x1bd4,0x53d4,0x000011af,0x0020fc,0x01,0x400,0x1f6e },
	{ 36,3,1,0,0x1fd4,0x57d4,0x00000131,0x00107e,0x02,0x400,0x2000 },
	{ 37,0,0,0,0x0353,0x114f,0x000131cd,0x000000,0x06,0x37f,0x0efb },
	{ 38,0,0,0,0x06d2,0x114f,0x000131cd,0x000000,0x06,0x37f,0x0efb },
	{ 39,0,0,0,0x0a51,0x114f,0x000131cd,0x000000,0x06,0x37f,0x0efb },
	{ 40,0,0,0,0x0dd0,0x114f,0x000131cd,0x000000,0x06,0x37f,0x0efb },
	{ 41,0,1,0,0x125f,0x125f,0x000065a5,0x00cc28,0x07,0x37f,0x1556 },
	{ 42,0,1,0,0x16ee,0x16ee,0x000008f9,0x005cac,0x08,0x37f,0x2000 },
	{ 43,0,0,0,0x1a6d,0x595b,0x000008f9,0x000000,0x00,0x37f,0x2000 },
	{ 44,0,0,0,0x1dec,0x595b,0x000008f9,0x000000,0x00,0x37f,0x2000 },
	{ 45,0,0,0,0x016b,0x395b,0x0008f9ef,0x000000,0x00,0x37f,0x0358 },
	{ 46,0,1,0,0x05fa,0x3a6b,0x0002252f,0x06d4c0,0x01,0x37f,0x0b9a },
	{ 47,0,0,0,0x0979,0x3a6b,0x0002252f,0x000000,0x01,0x37f,0x0b9a },
	{ 48,0,1,0,0x0e08,0x3b7b,0x0000741f,0x01b110,0x02,0x37f,0x1491 },
	{ 49,0,0,0,0x1187,0x3b7b,0x0000741f,0x000000,0x02,0x37f,0x1491 },
	{ 50,0,1,0,0x1616,0x3c8b,0x000008e3,0x006b3c,0x03,0x37f,0x2000 },
	{ 51,0,0,0,0x1995,0x3c8b,0x000008e3,0x000000,0x03,0x37f,0x2000 },
	{ 52,1,0,0,0x1d95,0x5195,0x000008e3,0x000000,0x01,0x400,0x2000 },
	{ 53,2,0,0,0x0195,0x3595,0x0008e301,0x000000,0x01,0x400,0x0366 },
	{ 54,3,1,0,0x0595,0x3995,0x0002cd41,0x0615c0,0x03,0x400,0x0a0f },
	{ 55,0,0,0,0x0914,0x288b,0x0002cd41,0x000000,0x03,0x37f,0x0a0f },
	{ 56,0,1,0,0x0da3,0x299b,0x0000fda1,0x01cfa0,0x04,0x37f,0x100f },
	{ 57,0,1,0,0x1232,0x2aab,0x00002b29,0x00d278,0x05,0x37f,0x1a47 },
	{ 58,0,0,0,0x15b1,0x2aab,0x00002b29,0x000000,0x05,0x37f,0x1a47 },
	{ 59,0,0,0,0x1930,0x2aab,0x00002b29,0x000000,0x05,0x37f,0x1a47 },
	{ 60,0,1,0,0x1dbf,0x2bbb,0x00000eba,0x001c6f,0x06,0x37f,0x2000 },
	{ 61,0,1,0,0x024e,0x0ccb,0x0001d223,0x0ce800,0x07,0x37f,0x0c8c },
	{ 62,0,0,0,0x05cd,0x0ccb,0x0001d223,0x000000,0x07,0x37f,0x0c8c },
	{ 63,0,0,0,0x094c,0x0ccb,0x0001d223,0x000000,0x07,0x37f,0x0c8c },
	{ 64,0,1,0,0x0ddb,0x0ddb,0x000013b3,0x01be70,0x08,0x37f,0x1ece },
	{ 65,0,0,0,0x115a,0x5048,0x000013b3,0x000000,0x00,0x37f,0x1ece },
	{ 66,0,0,0,0x14d9,0x5048,0x000013b3,0x000000,0x00,0x37f,0x1ece },
	{ 67,0,0,0,0x1858,0x5048,0x000013b3,0x000000,0x00,0x37f,0x1ece },
	{ 68,0,0,0,0x1bd7,0x5048,0x000013b3,0x000000,0x00,0x37f,0x1ece },
	{ 69,0,1,0,0x2066,0x5158,0x000001bc,0x0011f7,0x01,0x37f,0x2000 },
	{ 70,1,0,0,0x0466,0x3466,0x0001bc45,0x000000,0x01,0x400,0x0cd3 },
	{ 71,2,0,0,0x0866,0x3866,0x0001bc45,0x000000,0x01,0x400,0x0cd3 },
	{ 72,3,0,0,0x0c66,0x3c66,0x0001bc45,0x000000,0x03,0x400,0x0cd3 },
	{ 73,0,1,0,0x10f5,0x3e68,0x0000b775,0x0104d0,0x02,0x37f,0x11ee },
	{ 74,0,1,0,0x1584,0x3f78,0x00004115,0x007660,0x03,0x37f,0x17e8 },
	{ 75,0,1,0,0x1a13,0x4088,0x00000b5b,0x0035ba,0x04,0x37f,0x2000 },
	{ 76,0,0,0,0x1d92,0x4088,0x00000b5b,0x000000,0x04,0x37f,0x2000 },
	{ 77,0,0,0,0x0111,0x2088,0x000b5b67,0x000000,0x04,0x37f,0x01fc },
	{ 78,0,1,0,0x05a0,0x2198,0x00041927,0x074240,0x05,0x37f,0x07dd },
	{ 79,0,1,0,0x0a2f,0x22a8,0x0000cda7,0x034b80,0x06,0x37f,0x1145 },
	{ 80,0,0,0,0x0dae,0x22a8,0x0000cda7,0x000000,0x06,0x37f,0x1145 },
	{ 81,0,0,0,0x112d,0x22a8,0x0000cda7,0x000000,0x06,0x37f,0x1145 },
	{ 82,0,1,0,0x15bc,0x23b8,0x00005baf,0x0071f8,0x07,0x37f,0x15ee },
	{ 83,0,1,0,0x1a4b,0x24c8,0x000027f3,0x0033bc,0x08,0x37f,0x1aba },
	{ 84,0,1,0,0x1eda,0x25d8,0x00001078,0x00177b,0x09,0x37f,0x1fd6 },
	{ 85,0,1,0,0x0369,0x06e8,0x0005d009,0x0aa880,0x0a,0x37f,0x05d9 },
	{ 86,0,1,1,0x07f8,0x47f8,0x0000f989,0x04d680,0x00,0x400,0x1027 },
	{ 87,0,1,1,0x0bf8,0x47f8,0x0000f989,0x000000,0x00,0x400,0x1027 },
	{ 88,1,0,0,0x0ff8,0x3bf8,0x0000f989,0x000000,0x01,0x400,0x1027 },
	{ 89,2,1,0,0x13f8,0x3ff8,0x000078c9,0x0080c0,0x02,0x400,0x1457 },
	{ 90,3,1,0,0x17f8,0x43f8,0x00003869,0x004060,0x04,0x400,0x18bb },
	{ 91,0,0,1,0x1bf8,0x53f8,0x00001839,0x002030,0x01,0x400,0x1d9c },
	{ 92,0,0,1,0x1ff8,0x53f8,0x00000821,0x001018,0x02,0x400,0x2000 },
	{ 93,0,0,1,0x03f8,0x33f8,0x000015ab,0x080c00,0x03,0x400,0x1e41 },
	{ 94,0,1,1,0x07f8,0x33f8,0x000015ab,0x000000,0x03,0x400,0x1e41 },
	{ 95,0,1,1,0x0bf8,0x33f8,0x000015ab,0x000000,0x03,0x400,0x1e41 },
	{ 96,0,1,1,0x0ff8,0x33f8,0x000015ab,0x000000,0x03,0x400,0x1e41 },
	{ 97,0,1,1,0x13f8,0x33f8,0x000015ab,0x000000,0x03,0x400,0x1e41 },
	{ 98,0,1,1,0x17f8,0x33f8,0x000015ab,0x000000,0x03,0x400,0x1e41 },
	{ 99,0,1,1,0x1bf8,0x33f8,0x000015ab,0x000000,0x03,0x400,0x1e41 },
	{ 100,0,0,1,0x1ff8,0x33f8,0x00000593,0x001018,0x04,0x400,0x2000 },
	{ 101,0,1,1,0x03f8,0x13f8,0x000593cd,0x000000,0x04,0x400,0x0616 },
	{ 102,0,0,1,0x07f8,0x13f8,0x00018dcd,0x040600,0x05,0x400,0x0d76 },
	{ 103,0,1,1,0x0bf8,0x13f8,0x00018dcd,0x000000,0x05,0x400,0x0d76 },
	{ 104,0,0,1,0x0ff8,0x13f8,0x00008c4d,0x010180,0x06,0x400,0x137a },
	{ 105,0,0,1,0x13f8,0x13f8,0x00000b8d,0x0080c0,0x07,0x400,0x2000 },
	{ 106,1,0,0,0x17f8,0x3ff8,0x00000b8d,0x000000,0x01,0x400,0x2000 },
	{ 107,2,0,0,0x1bf8,0x43f8,0x00000b8d,0x000000,0x02,0x400,0x2000 },
	{ 108,3,0,0,0x1ff8,0x47f8,0x00000b8d,0x000000,0x04,0x400,0x2000 },
	{ 109,0,0,1,0x03f8,0x03f8,0x000381ef,0x080c00,0x08,0x400,0x08c3 },
	{ 110,0,1,1,0x07f8,0x47f8,0x000381ef,0x000000,0x00,0x400,0x08c3 },
	{ 111,0,0,1,0x0bf8,0x47f8,0x00017eef,0x020300,0x01,0x400,0x0dae },
	{ 112,0,0,1,0x0ff8,0x47f8,0x00007d6f,0x010180,0x02,0x400,0x141f },
	{ 113,0,1,1,0x13f8,0x47f8,0x00007d6f,0x000000,0x02,0x400,0x141f },
	{ 114,0,0,1,0x17f8,0x47f8,0x00003d0f,0x004060,0x03,0x400,0x1846 },
	{ 115,0,0,1,0x1bf8,0x47f8,0x00001cdf,0x002030,0x04,0x400,0x1c99 },
	{ 116,0,0,1,0x1ff8,0x47f8,0x00000cc7,0x001018,0x05,0x400,0x2000 },
	{ 117,0,0,1,0x03f8,0x27f8,0x0004bb01,0x080c00,0x06,0x400,0x070a },
	{ 118,0,0,1,0x07f8,0x27f8,0x0000b501,0x040600,0x07,0x400,0x1201 },
	{ 119,0,1,1,0x0bf8,0x27f8,0x0000b501,0x000000,0x07,0x400,0x1201 },
	{ 120,0,1,1,0x0ff8,0x27f8,0x0000b501,0x000000,0x07,0x400,0x1201 },
	{ 121,0,0,1,0x13f8,0x27f8,0x00003441,0x0080c0,0x08,0x400,0x192d },
	{ 122,0,1,1,0x17f8,0x27f8,0x00003441,0x000000,0x08,0x400,0x192d },
	{ 123,0,0,1,0x1bf8,0x27f8,0x00001411,0x002030,0x09,0x400,0x1eb3 },
	{ 124,1,1,0,0x1ff8,0x43f8,0x000003f9,0x001018,0x02,0x400,0x2000 },
	{ 125,2,0,0,0x03f8,0x27f8,0x0003f923,0x000000,0x02,0x400,0x080b },
	{ 126,3,0,0,0x07f8,0x2bf8,0x0003f923,0x000000,0x04,0x400,0x080b },
	{ 127,0,0,1,0x0bf8,0x13f8,0x0001f623,0x020300,0x0a,0x400,0x0c1e },
	{ 128,0,0,0,0x0ff8,0x4ee6,0x0000f4a3,0x010180,0x00,0x37f,0x1044 },
	{ 129,0,1,0,0x1487,0x4ff6,0x0000681b,0x008c88,0x01,0x37f,0x1532 },
	{ 130,0,1,0,0x1916,0x5106,0x00002859,0x003fc2,0x02,0x37f,0x1aab },
	{ 131,0,1,0,0x1da5,0x5216,0x00000b69,0x001cf0,0x03,0x37f,0x2000 },
	{ 132,0,0,0,0x0124,0x3216,0x000b6945,0x000000,0x03,0x37f,0x01f5 },
	{ 133,0,1,0,0x05b3,0x3326,0x00043e85,0x072ac0,0x04,0x37f,0x07aa },
	{ 134,0,1,0,0x0a42,0x3436,0x0000fdc5,0x0340c0,0x05,0x37f,0x100e },
	{ 135,0,0,0,0x0dc1,0x3436,0x0000fdc5,0x000000,0x05,0x37f,0x100e },
	{ 136,0,1,0,0x1250,0x3546,0x00002f8d,0x00ce38,0x06,0x37f,0x19b8 },
	{ 137,0,0,0,0x15cf,0x3546,0x00002f8d,0x000000,0x06,0x37f,0x19b8 },
	{ 138,0,0,0,0x194e,0x3546,0x00002f8d,0x000000,0x06,0x37f,0x19b8 },
	{ 139,0,1,0,0x1ddd,0x3656,0x000013b0,0x001bdd,0x07,0x37f,0x1ecf },
	{ 140,0,1,0,0x026c,0x1766,0x00070a67,0x0ca600,0x08,0x37f,0x04be },
	{ 141,0,1,0,0x06fb,0x1876,0x00014ca7,0x05bdc0,0x09,0x37f,0x0e7e },
	{ 142,1,0,0,0x0afb,0x2afb,0x00014ca7,0x000000,0x02,0x400,0x0e7e },
	{ 143,2,1,0,0x0efb,0x2efb,0x00001af7,0x0131b0,0x03,0x400,0x1cfe },
	{ 144,3,0,0,0x12fb,0x32fb,0x00001af7,0x000000,0x04,0x400,0x1cfe },
	{ 145,0,0,0,0x167a,0x2476,0x00001af7,0x000000,0x09,0x37f,0x1cfe },
	{ 146,0,0,0,0x19f9,0x2476,0x00001af7,0x000000,0x09,0x37f,0x1cfe },
	{ 147,0,1,0,0x1e88,0x2586,0x00000225,0x0018d2,0x0a,0x37f,0x2000 },
	{ 148,0,0,0,0x0207,0x0586,0x00022589,0x000000,0x0a,0x37f,0x0b99 },
	{ 149,0,0,0,0x0586,0x4474,0x00022589,0x000000,0x00,0x37f,0x0b99 },
	{ 150,0,0,0,0x0905,0x4474,0x00022589,0x000000,0x00,0x37f,0x0b99 },
	{ 151,0,1,0,0x0d94,0x4584,0x00005129,0x01d460,0x01,0x37f,0x16a2 },
	{ 152,0,0,0,0x1113,0x4584,0x00005129,0x000000,0x01,0x37f,0x16a2 },
	{ 153,0,0,0,0x1492,0x4584,0x00005129,0x000000,0x01,0x37f,0x16a2 },
	{ 154,0,1,0,0x1921,0x4694,0x000011e1,0x003f48,0x02,0x37f,0x1f5d },
	{ 155,0,0,0,0x1ca0,0x4694,0x000011e1,0x000000,0x02,0x37f,0x1f5d },
	{ 156,0,1,0,0x012f,0x27a4,0x000235ab,0x0fac00,0x03,0x37f,0x0b6e },
	{ 157,0,0,0,0x04ae,0x27a4,0x000235ab,0x000000,0x03,0x37f,0x0b6e },
	{ 158,0,0,0,0x082d,0x27a4,0x000235ab,0x000000,0x03,0x37f,0x0b6e },
	{ 159,0,1,0,0x0cbc,0x28b4,0x0000176b,0x021e40,0x04,0x37f,0x1dce },
	{ 160,1,0,0,0x10bc,0x2cbc,0x0000176b,0x000000,0x02,0x400,0x1dce },
	{ 161,2,0,0,0x14bc,0x30bc,0x0000176b,0x000000,0x03,0x400,0x1dce },
	{ 162,3,0,0,0x18bc,0x34bc,0x0000176b,0x000000,0x04,0x400,0x1dce },
	{ 163,0,0,0,0x1c3b,0x34b4,0x0000176b,0x000000,0x04,0x37f,0x1dce },
	{ 164,0,1,0,0x20ca,0x35c4,0x000006a2,0x0010c9,0x05,0x37f,0x2000 },
	{ 165,0,0,0,0x0449,0x15c4,0x0006a2cd,0x000000,0x05,0x37f,0x0515 },
	{ 166,0,1,0,0x08d8,0x16d4,0x00027acd,0x042800,0x06,0x37f,0x0ac4 },
	{ 167,0,1,0,0x0d67,0x17e4,0x000097ed,0x01e2e0,0x07,0x37f,0x1304 },
	{ 168,0,0,0,0x10e6,0x17e4,0x000097ed,0x000000,0x07,0x37f,0x1304 },
	{ 169,0,1,0,0x1575,0x18f4,0x00002059,0x007794,0x08,0x37f,0x1bf1 },
	{ 170,0,0,0,0x18f4,0x57e2,0x00002059,0x000000,0x00,0x37f,0x1bf1 },
	{ 171,0,1,0,0x1d83,0x58f2,0x000002bc,0x001d9d,0x01,0x37f,0x2000 },
	{ 172,0,0,0,0x0102,0x38f2,0x0002bcef,0x000000,0x01,0x37f,0x0a31 },
	{ 173,0,0,0,0x0481,0x38f2,0x0002bcef,0x000000,0x01,0x37f,0x0a31 },
	{ 174,0,0,0,0x0800,0x38f2,0x0002bcef,0x000000,0x01,0x37f,0x0a31 },
	{ 175,0,1,0,0x0c8f,0x3a02,0x00008dcf,0x022f20,0x02,0x37f,0x136a },
	{ 176,0,0,0,0x100e,0x3a02,0x00008dcf,0x000000,0x02,0x37f,0x136a },
	{ 177,0,1,0,0x149d,0x3b12,0x00000357,0x008a78,0x03,0x37f,0x2000 },
	{ 178,1,0,0,0x189d,0x309d,0x00000357,0x000000,0x02,0x400,0x2000 },
	{ 179,2,0,0,0x1c9d,0x349d,0x00000357,0x000000,0x03,0x400,0x2000 },
	{ 180,3,0,0,0x009d,0x189d,0x00035701,0x000000,0x04,0x400,0x090b },
	{ 181,0,0,0,0x041c,0x2712,0x00035701,0x000000,0x03,0x37f,0x090b },
	{ 182,0,0,0,0x079b,0x2712,0x00035701,0x000000,0x03,0x37f,0x090b },
	{ 183,0,1,0,0x0c2a,0x2822,0x00010061,0x0256a0,0x04,0x37f,0x0fff },
	{ 184,0,0,0,0x0fa9,0x2822,0x00010061,0x000000,0x04,0x37f,0x0fff },
	{ 185,0,1,0,0x1438,0x2932,0x00006c21,0x009440,0x05,0x37f,0x14fa },
	{ 186,0,1,0,0x18c7,0x2a42,0x000028d9,0x004348,0x06,0x37f,0x1a99 },
	{ 187,0,1,0,0x1d56,0x2b52,0x00000a52,0x001e87,0x07,0x37f,0x2000 },
	{ 188,0,0,0,0x00d5,0x0b52,0x000a5223,0x000000,0x07,0x37f,0x0289 },
	{ 189,0,1,0,0x0564,0x0c62,0x0002c2e3,0x078f40,0x08,0x37f,0x0a25 },
	{ 190,0,0,0,0x08e3,0x0c62,0x0002c2e3,0x000000,0x08,0x37f,0x0a25 },
	{ 191,0,1,0,0x0d72,0x0d72,0x0000e3a3,0x01df40,0x09,0x37f,0x10ae },
	{ 192,0,1,0,0x1201,0x1201,0x00000a13,0x00d990,0x0a,0x37f,0x2000 },
	{ 193,0,0,0,0x1580,0x546e,0x00000a13,0x000000,0x00,0x37f,0x2000 },
	{ 194,0,0,0,0x18ff,0x546e,0x00000a13,0x000000,0x00,0x37f,0x2000 },
	{ 195,0,0,0,0x1c7e,0x546e,0x00000a13,0x000000,0x00,0x37f,0x2000 },
	{ 196,1,0,0,0x007e,0x147e,0x000a1345,0x000000,0x02,0x400,0x02ad },
	{ 197,2,1,0,0x047e,0x187e,0x0002ba05,0x075940,0x04,0x400,0x0a37 },
	{ 198,3,0,0,0x087e,0x1c7e,0x0002ba05,0x000000,0x04,0x400,0x0a37 },
	{ 199,0,1,0,0x0d0d,0x417e,0x0000b8c5,0x020140,0x01,0x37f,0x11e3 },
	{ 200,0,0,0,0x108c,0x417e,0x0000b8c5,0x000000,0x01,0x37f,0x11e3 },
	{ 201,0,1,0,0x151b,0x428e,0x000039b1,0x007f14,0x02,0x37f,0x189a },
	{ 202,0,1,0,0x19aa,0x439e,0x00000003,0x0039ae,0x03,0x37f,0x2000 },
	{ 203,0,0,0,0x1d29,0x439e,0x00000003,0x000000,0x03,0x37f,0x2000 },
	{ 204,0,0,0,0x00a8,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 205,0,0,0,0x0427,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 206,0,0,0,0x07a6,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 207,0,0,0,0x0b25,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 208,0,0,0,0x0ea4,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 209,0,0,0,0x1223,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 210,0,0,0,0x15a2,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 211,0,0,0,0x1921,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 212,0,0,0,0x1ca0,0x239e,0x00000367,0x000000,0x03,0x37f,0x2000 },
	{ 213,0,0,0,0x001f,0x039e,0x00036789,0x000000,0x03,0x37f,0x08ef },
	{ 214,1,0,0,0x041f,0x141f,0x00036789,0x000000,0x02,0x400,0x08ef },
	{ 215,2,0,0,0x081f,0x181f,0x00036789,0x000000,0x04,0x400,0x08ef },
	{ 216,3,1,0,0x0c1f,0x1c1f,0x00017209,0x01f580,0x05,0x400,0x0de1 },
	{ 217,0,1,0,0x10ae,0x10ae,0x00006049,0x0111c0,0x04,0x37f,0x15a6 },
	{ 218,0,0,0,0x142d,0x4f2e,0x00006049,0x000000,0x00,0x31b,0x15a6 },
	{ 219,0,1,0,0x193e,0x5124,0x000017c1,0x004888,0x01,0x31b,0x1db9 },
	{ 220,0,0,0,0x1c59,0x5124,0x000017c1,0x000000,0x01,0x31b,0x1db9 },
	{ 221,0,1,0,0x216a,0x531a,0x00000627,0x00119a,0x02,0x31b,0x2000 },
	{ 222,0,0,0,0x0485,0x331a,0x000627ab,0x000000,0x02,0x31b,0x0585 },
	{ 223,0,1,0,0x0996,0x3510,0x0001e22b,0x044580,0x03,0x31b,0x0c5a },
	{ 224,0,1,0,0x0ea7,0x3706,0x00001bdb,0x01c650,0x04,0x31b,0x1cce },
	{ 225,0,0,0,0x11c2,0x3706,0x00001bdb,0x000000,0x04,0x31b,0x1cce },
	{ 226,0,0,0,0x14dd,0x3706,0x00001bdb,0x000000,0x04,0x31b,0x1cce },
	{ 227,0,0,0,0x17f8,0x3706,0x00001bdb,0x000000,0x04,0x31b,0x1cce },
	{ 228,0,0,0,0x1b13,0x3706,0x00001bdb,0x000000,0x04,0x31b,0x1cce },
	{ 229,0,1,0,0x2024,0x38fc,0x000005e9,0x0015f2,0x05,0x31b,0x2000 },
	{ 230,0,0,0,0x033f,0x18fc,0x0005e9cd,0x000000,0x05,0x31b,0x05c0 },
	{ 231,0,1,0,0x0850,0x1af2,0x0000964d,0x055380,0x06,0x31b,0x1314 },
	{ 232,1,0,0,0x0c50,0x1850,0x0000964d,0x000000,0x02,0x400,0x1314 },
	{ 233,2,0,0,0x1050,0x1c50,0x0000964d,0x000000,0x04,0x400,0x1314 },
	{ 234,3,1,0,0x1450,0x2050,0x00001d05,0x007948,0x06,0x400,0x1c91 },
	{ 235,0,0,0,0x176b,0x26f2,0x00001d05,0x000000,0x06,0x31b,0x1c91 },
	{ 236,0,0,0,0x1a86,0x26f2,0x00001d05,0x000000,0x06,0x31b,0x1c91 },
	{ 237,0,1,0,0x1f97,0x28e8,0x000004e1,0x001824,0x07,0x31b,0x2000 },
	{ 238,0,0,0,0x02b2,0x08e8,0x0004e1ef,0x000000,0x07,0x31b,0x06db },
	{ 239,0,0,0,0x05cd,0x08e8,0x0004e1ef,0x000000,0x07,0x31b,0x06db },
	{ 240,0,1,0,0x0ade,0x0ade,0x0001766f,0x036b80,0x08,0x31b,0x0dd0 },
	{ 241,0,1,0,0x0fef,0x0fef,0x00000a7f,0x016bf0,0x09,0x31b,0x2000 },
	{ 242,0,0,0,0x130a,0x4e0b,0x00000a7f,0x000000,0x00,0x31b,0x2000 },
	{ 243,0,0,0,0x1625,0x4e0b,0x00000a7f,0x000000,0x00,0x31b,0x2000 },
	{ 244,0,0,0,0x1940,0x4e0b,0x00000a7f,0x000000,0x00,0x31b,0x2000 },
	{ 245,0,0,0,0x1c5b,0x4e0b,0x00000a7f,0x000000,0x00,0x31b,0x2000 },
	{ 246,0,0,0,0x1f76,0x4e0b,0x00000a7f,0x000000,0x00,0x31b,0x2000 },
	{ 247,0,1,0,0x0487,0x3001,0x00003c01,0x0a4300,0x01,0x31b,0x1860 },
	{ 248,0,0,0,0x07a2,0x3001,0x00003c01,0x000000,0x01,0x31b,0x1860 },
	{ 249,0,0,0,0x0abd,0x3001,0x00003c01,0x000000,0x01,0x31b,0x1860 },
	{ 250,1,0,0,0x0ebd,0x16bd,0x00003c01,0x000000,0x02,0x400,0x1860 },
	{ 251,2,0,0,0x12bd,0x1abd,0x00003c01,0x000000,0x04,0x400,0x1860 },
	{ 252,3,0,0,0x16bd,0x1ebd,0x00003c01,0x000000,0x06,0x400,0x1860 },
	{ 253,0,1,0,0x1bce,0x3df7,0x00000d7b,0x002e86,0x02,0x31b,0x2000 },
	{ 254,0,0,0,0x1ee9,0x3df7,0x00000d7b,0x000000,0x02,0x31b,0x2000 },
	{ 255,0,1,0,0x03fa,0x1fed,0x00023123,0x0b4a00,0x03,0x31b,0x0b7a },
	{ 256,0,0,0,0x0715,0x1fed,0x00023123,0x000000,0x03,0x31b,0x0b7a },
	{ 257,0,0,0,0x0a30,0x1fed,0x00023123,0x000000,0x03,0x31b,0x0b7a },
	{ 258,0,1,0,0x0f41,0x21e3,0x000097c3,0x019960,0x04,0x31b,0x1306 },
	{ 259,0,0,0,0x125c,0x21e3,0x000097c3,0x000000,0x04,0x31b,0x1306 },
	{ 260,0,1,0,0x176d,0x23d9,0x0000346b,0x006358,0x05,0x31b,0x1928 },
	{ 261,0,1,0,0x1c7e,0x25cf,0x00000b1f,0x00294c,0x06,0x31b,0x2000 },
	{ 262,0,0,0,0x1f99,0x25cf,0x00000b1f,0x000000,0x06,0x31b,0x2000 },
	{ 263,0,1,0,0x04aa,0x07c5,0x000119c5,0x0a0580,0x07,0x31b,0x0f74 },
	{ 264,0,0,0,0x07c5,0x42c6,0x000119c5,0x000000,0x00,0x31b,0x0f74 },
	{ 265,0,0,0,0x0ae0,0x42c6,0x000119c5,0x000000,0x00,0x31b,0x0f74 },
	{ 266,0,0,0,0x0dfb,0x42c6,0x000119c5,0x000000,0x00,0x31b,0x0f74 },
	{ 267,0,1,0,0x130c,0x44bc,0x0000459d,0x00d428,0x01,0x31b,0x1785 },
	{ 268,1,0,0,0x170c,0x1b0c,0x0000459d,0x000000,0x02,0x400,0x1785 },
	{ 269,2,1,0,0x1b0c,0x1f0c,0x00001fd7,0x0025c6,0x05,0x400,0x1c08 },
	{ 270,3,1,0,0x1f0c,0x230c,0x00000cf4,0x0012e3,0x07,0x400,0x2000 },
	{ 271,0,1,0,0x041d,0x32b2,0x0001ede7,0x0b0680,0x02,0x31b,0x0c36 },
	{ 272,0,0,0,0x0738,0x32b2,0x0001ede7,0x000000,0x02,0x31b,0x0c36 },
	{ 273,0,0,0,0x0a53,0x32b2,0x0001ede7,0x000000,0x02,0x31b,0x0c36 },
	{ 274,0,1,0,0x0f64,0x34a8,0x00005e17,0x018fd0,0x03,0x31b,0x15c8 },
	{ 275,0,0,0,0x127f,0x34a8,0x00005e17,0x000000,0x03,0x31b,0x15c8 },
	{ 276,0,0,0,0x159a,0x34a8,0x00005e17,0x000000,0x03,0x31b,0x15c8 },
	{ 277,0,1,0,0x1aab,0x369e,0x00002573,0x0038a4,0x04,0x31b,0x1b19 },
	{ 278,0,1,0,0x1fbc,0x3894,0x00000de7,0x00178c,0x05,0x31b,0x2000 },
	{ 279,0,1,0,0x04cd,0x1a8a,0x00041e09,0x09c980,0x06,0x31b,0x07d6 },
	{ 280,0,1,0,0x09de,0x1c80,0x00000cc9,0x041140,0x07,0x31b,0x2000 },
	{ 281,0,0,0,0x0cf9,0x1c80,0x00000cc9,0x000000,0x07,0x31b,0x2000 },
	{ 282,0,0,0,0x1014,0x1c80,0x00000cc9,0x000000,0x07,0x31b,0x2000 },
	{ 283,0,0,0,0x132f,0x1c80,0x00000cc9,0x000000,0x07,0x31b,0x2000 },
	{ 284,0,0,0,0x164a,0x1c80,0x00000cc9,0x000000,0x07,0x31b,0x2000 },
	{ 285,0,0,0,0x1965,0x1c80,0x00000cc9,0x000000,0x07,0x31b,0x2000 },
	{ 286,1,0,0,0x1d65,0x5c53,0x00000cc9,0x000000,0x00,0x37f,0x2000 },
	{ 287,2,1,0,0x0165,0x0165,0x000037ab,0x0c9200,0x06,0x400,0x18cf },
	{ 288,3,0,0,0x0565,0x4565,0x000037ab,0x000000,0x00,0x400,0x18cf },
	{ 289,0,0,0,0x0880,0x4381,0x000037ab,0x000000,0x00,0x31b,0x18cf },
	{ 290,0,0,0,0x0b9b,0x4381,0x000037ab,0x000000,0x00,0x31b,0x18cf },
	{ 291,0,0,0,0x0eb6,0x4381,0x000037ab,0x000000,0x00,0x31b,0x18cf },
	{ 292,0,0,0,0x11d1,0x4381,0x000037ab,0x000000,0x00,0x31b,0x18cf },
	{ 293,0,0,0,0x14ec,0x4381,0x000037ab,0x000000,0x00,0x31b,0x18cf },
	{ 294,0,0,0,0x1807,0x4381,0x000037ab,0x000000,0x00,0x31b,0x18cf },
	{ 295,0,1,0,0x1d18,0x4577,0x00001275,0x002536,0x01,0x31b,0x1f2e },
	{ 296,0,1,0,0x0229,0x276d,0x0002ffcd,0x0f7600,0x02,0x31b,0x09ab },
};

uint8_t stream7[]={ 0x65,0x43,0x2a,0x1e,0x04,0x65,0x44,0x9f,0x25,0x23,0x17,0xbc,0xa3,0xb9,0xd1,0x09,0x57,0x4a,0x6f,0x33,0x5b,0xa3,0xe8,0xd0,0x38,0x94,0x13,0x15,0x45,0xfd,0xfc,0x67,0x0a,0x18,0x2a,0x1b,0xa4,0x18,0xc7,0x7c,0xde,0xe3,0x17,0x3d,0x88,0xe5,0x07,0x85,0x93,0x05,0xeb,0xf3,0xfe,0x43,0x61,0x2c,0x31,0x23,0xfe,0x62,0xa7,0xf8,0x0e,0x83,0x1b,0xd1,0x7a,0x5c,0x07,0xe2,0x53,0x2b,0x8f,0x8f,0x39,0xa9,0x60,0x95,0xdb,0x07,0x1d,0x26,0xed,0x32,0xa8,0xad,0x39,0x6a,0x0d,0x2b,0x8e,0x3c,0x0f,0xe4,0x90,0x55, };

struct KnownState knownstates7[]=
{
//	 i,s,yn,mps,  lr,   lrm,         x,      dx,   k,   lp,    lx
	{ 0,-1,-1,-1,0x1001,0x1001,0x00006543,-1,-1,-1,-1 },
	{ 1,0,0,0,0x1401,0x5001,0x00006543,0x000000,0x00,0x400,0x155b },
	{ 2,0,1,0,0x1801,0x5001,0x00002549,0x003ffa,0x01,0x400,0x1b20 },
	{ 3,0,1,0,0x1c01,0x5001,0x0000054c,0x001ffd,0x02,0x400,0x2000 },
	{ 4,0,0,0,0x0001,0x3001,0x00054c2a,0x000000,0x02,0x400,0x0662 },
	{ 5,0,0,0,0x0401,0x3001,0x00054c2a,0x000000,0x02,0x400,0x0662 },
	{ 6,0,1,0,0x0801,0x3001,0x00014c8a,0x03ffa0,0x03,0x400,0x0e7f },
	{ 7,0,0,0,0x0c01,0x3001,0x00014c8a,0x000000,0x03,0x400,0x0e7f },
	{ 8,0,1,0,0x1001,0x3001,0x00004ca2,0x00ffe8,0x04,0x400,0x16f7 },
	{ 9,0,0,0,0x1401,0x3001,0x00004ca2,0x000000,0x04,0x400,0x16f7 },
	{ 10,0,1,0,0x1801,0x3001,0x00000ca8,0x003ffa,0x05,0x400,0x2000 },
	{ 11,0,0,0,0x1c01,0x3001,0x00000ca8,0x000000,0x05,0x400,0x2000 },
	{ 12,0,0,0,0x0001,0x1001,0x000ca81e,0x000000,0x05,0x400,0x015b },
	{ 13,0,1,0,0x0401,0x1001,0x0004a8de,0x07ff40,0x06,0x400,0x0720 },
	{ 14,0,1,0,0x0801,0x1001,0x0000a93e,0x03ffa0,0x07,0x400,0x1265 },
	{ 15,0,0,0,0x0c01,0x1001,0x0000a93e,0x000000,0x07,0x400,0x1265 },
	{ 16,0,0,0,0x1001,0x5001,0x0000a93e,0x000000,0x00,0x400,0x1265 },
	{ 17,0,1,0,0x1401,0x5001,0x0000294a,0x007ff4,0x01,0x400,0x1a89 },
	{ 18,0,0,0,0x1801,0x5001,0x0000294a,0x000000,0x01,0x400,0x1a89 },
	{ 19,0,1,0,0x1c01,0x5001,0x0000094d,0x001ffd,0x02,0x400,0x2000 },
	{ 20,0,0,0,0x0001,0x3001,0x00094d04,0x000000,0x02,0x400,0x0323 },
	{ 21,0,1,0,0x0401,0x3001,0x00014dc4,0x07ff40,0x03,0x400,0x0e79 },
	{ 22,0,0,0,0x0801,0x3001,0x00014dc4,0x000000,0x03,0x400,0x0e79 },
	{ 23,0,0,0,0x0c01,0x3001,0x00014dc4,0x000000,0x03,0x400,0x0e79 },
	{ 24,0,1,0,0x1001,0x3001,0x00004ddc,0x00ffe8,0x04,0x400,0x16e0 },
	{ 25,0,0,0,0x1401,0x3001,0x00004ddc,0x000000,0x04,0x400,0x16e0 },
	{ 26,0,1,0,0x1801,0x3001,0x00000de2,0x003ffa,0x05,0x400,0x2000 },
	{ 27,0,0,0,0x1c01,0x3001,0x00000de2,0x000000,0x05,0x400,0x2000 },
	{ 28,0,0,0,0x0001,0x1001,0x000de265,0x000000,0x05,0x400,0x00d3 },
	{ 29,0,1,0,0x0401,0x1001,0x0005e325,0x07ff40,0x06,0x400,0x05c6 },
	{ 30,0,1,0,0x0801,0x1001,0x0001e385,0x03ffa0,0x07,0x400,0x0c55 },
	{ 31,0,0,0,0x0c01,0x1001,0x0001e385,0x000000,0x07,0x400,0x0c55 },
	{ 32,0,1,0,0x1001,0x1001,0x0000e39d,0x00ffe8,0x08,0x400,0x10af },
	{ 33,0,1,0,0x1401,0x1401,0x000063a9,0x007ff4,0x09,0x400,0x1573 },
	{ 34,0,1,0,0x1801,0x1801,0x000023af,0x003ffa,0x0a,0x400,0x1b61 },
	{ 35,0,1,1,0x1c01,0x5aef,0x000003b2,0x001ffd,0x00,0x37f,0x2000 },
	{ 36,0,1,1,0x1f80,0x5aef,0x000003b2,0x000000,0x00,0x37f,0x2000 },
	{ 37,0,1,1,0x02ff,0x3aef,0x0003b244,0x000000,0x00,0x37f,0x0876 },
	{ 38,0,1,1,0x067e,0x3aef,0x0003b244,0x000000,0x00,0x37f,0x0876 },
	{ 39,0,0,1,0x0b0d,0x3bff,0x0000dc44,0x02d600,0x01,0x37f,0x10df },
	{ 40,0,1,1,0x0e8c,0x3bff,0x0000dc44,0x000000,0x01,0x37f,0x10df },
	{ 41,0,0,1,0x131b,0x3d0f,0x0000287c,0x00b3c8,0x02,0x37f,0x1aa6 },
	{ 42,0,1,1,0x169a,0x3d0f,0x0000287c,0x000000,0x02,0x37f,0x1aa6 },
	{ 43,0,1,1,0x1a19,0x3d0f,0x0000287c,0x000000,0x02,0x37f,0x1aa6 },
	{ 44,0,0,1,0x1ea8,0x3e1f,0x00001032,0x00184a,0x03,0x37f,0x1fef },
	{ 45,0,0,1,0x0337,0x1f2f,0x00052c1f,0x0b0680,0x04,0x37f,0x0686 },
	{ 46,0,0,1,0x07c6,0x203f,0x00002adf,0x050140,0x05,0x37f,0x1a51 },
	{ 47,0,1,1,0x0b45,0x203f,0x00002adf,0x000000,0x05,0x37f,0x1a51 },
	{ 48,0,1,1,0x0ec4,0x203f,0x00002adf,0x000000,0x05,0x37f,0x1a51 },
	{ 49,0,1,1,0x1243,0x203f,0x00002adf,0x000000,0x05,0x37f,0x1a51 },
	{ 50,0,1,1,0x15c2,0x203f,0x00002adf,0x000000,0x05,0x37f,0x1a51 },
	{ 51,0,1,1,0x1941,0x203f,0x00002adf,0x000000,0x05,0x37f,0x1a51 },
	{ 52,0,0,1,0x1dd0,0x214f,0x00000ec3,0x001c1c,0x06,0x37f,0x2000 },
	{ 53,0,0,1,0x025f,0x025f,0x000200a5,0x0cc280,0x07,0x37f,0x0bff },
	{ 54,0,1,1,0x05de,0x44cc,0x000200a5,0x000000,0x00,0x37f,0x0bff },
	{ 55,0,1,1,0x095d,0x44cc,0x000200a5,0x000000,0x00,0x37f,0x0bff },
	{ 56,0,0,1,0x0dec,0x45dc,0x00004755,0x01b950,0x01,0x37f,0x1761 },
	{ 57,0,1,1,0x116b,0x45dc,0x00004755,0x000000,0x01,0x37f,0x1761 },
	{ 58,0,1,1,0x14ea,0x45dc,0x00004755,0x000000,0x01,0x37f,0x1761 },
	{ 59,0,0,1,0x1979,0x46ec,0x00000bb5,0x003ba0,0x02,0x37f,0x2000 },
	{ 60,0,1,1,0x1cf8,0x46ec,0x00000bb5,0x000000,0x02,0x37f,0x2000 },
	{ 61,0,1,1,0x0077,0x26ec,0x000bb523,0x000000,0x02,0x37f,0x01cf },
	{ 62,0,0,1,0x0506,0x27fc,0x0003a6a3,0x080e80,0x03,0x37f,0x0888 },
	{ 63,0,1,1,0x0885,0x27fc,0x0003a6a3,0x000000,0x03,0x37f,0x0888 },
	{ 64,0,0,1,0x0d14,0x290c,0x0001a7e3,0x01fec0,0x04,0x37f,0x0d18 },
	{ 65,0,0,1,0x11a3,0x2a1c,0x0000c013,0x00e7d0,0x05,0x37f,0x11aa },
	{ 66,0,0,1,0x1632,0x2b2c,0x000056d7,0x00693c,0x06,0x37f,0x163f },
	{ 67,0,0,1,0x1ac1,0x2c3c,0x00002713,0x002fc4,0x07,0x37f,0x1ada },
	{ 68,0,0,1,0x1f50,0x2d4c,0x00001165,0x0015ae,0x08,0x37f,0x1f86 },
	{ 69,0,0,1,0x03df,0x0e5c,0x00078e17,0x09d700,0x09,0x37f,0x0455 },
	{ 70,0,0,1,0x086e,0x0f6c,0x00031697,0x047780,0x0a,0x37f,0x097f },
	{ 71,0,0,0,0x0cfd,0x4cfd,0x00010fb7,0x0206e0,0x00,0x400,0x0fa9 },
	{ 72,0,1,0,0x10fd,0x4cfd,0x000037ef,0x00d7c8,0x01,0x400,0x18c8 },
	{ 73,0,0,0,0x14fd,0x4cfd,0x000037ef,0x000000,0x01,0x400,0x18c8 },
	{ 74,0,1,0,0x18fd,0x4cfd,0x000001fd,0x0035f2,0x02,0x400,0x2000 },
	{ 75,0,0,0,0x1cfd,0x4cfd,0x000001fd,0x000000,0x02,0x400,0x2000 },
	{ 76,0,0,0,0x00fd,0x2cfd,0x0001fdbc,0x000000,0x02,0x400,0x0c07 },
	{ 77,0,0,0,0x04fd,0x2cfd,0x0001fdbc,0x000000,0x02,0x400,0x0c07 },
	{ 78,0,0,0,0x08fd,0x2cfd,0x0001fdbc,0x000000,0x02,0x400,0x0c07 },
	{ 79,0,1,0,0x0cfd,0x2cfd,0x00004e2c,0x01af90,0x03,0x400,0x16da },
	{ 80,0,0,0,0x10fd,0x2cfd,0x00004e2c,0x000000,0x03,0x400,0x16da },
	{ 81,0,0,0,0x14fd,0x2cfd,0x00004e2c,0x000000,0x03,0x400,0x16da },
	{ 82,0,1,0,0x18fd,0x2cfd,0x0000183a,0x0035f2,0x04,0x400,0x1d9c },
	{ 83,0,0,0,0x1cfd,0x2cfd,0x0000183a,0x000000,0x04,0x400,0x1d9c },
	{ 84,0,1,0,0x00fd,0x0cfd,0x000abe23,0x0d7c80,0x05,0x400,0x024e },
	{ 85,0,1,0,0x04fd,0x0cfd,0x0003ffe3,0x06be40,0x06,0x400,0x0801 },
	{ 86,0,1,0,0x08fd,0x0cfd,0x0000a0c3,0x035f20,0x07,0x400,0x12b1 },
	{ 87,0,0,0,0x0cfd,0x4cfd,0x0000a0c3,0x000000,0x00,0x400,0x12b1 },
	{ 88,0,0,0,0x10fd,0x4cfd,0x0000a0c3,0x000000,0x00,0x400,0x12b1 },
	{ 89,0,1,0,0x14fd,0x4cfd,0x000034df,0x006be4,0x01,0x400,0x191b },
	{ 90,0,0,0,0x18fd,0x4cfd,0x000034df,0x000000,0x01,0x400,0x191b },
	{ 91,0,1,0,0x1cfd,0x4cfd,0x000019e6,0x001af9,0x02,0x400,0x1d3a },
	{ 92,0,1,0,0x00fd,0x2cfd,0x000c6a39,0x0d7c80,0x03,0x400,0x0178 },
	{ 93,0,1,0,0x04fd,0x2cfd,0x0005abf9,0x06be40,0x04,0x400,0x05fe },
	{ 94,0,1,0,0x08fd,0x2cfd,0x00024cd9,0x035f20,0x05,0x400,0x0b33 },
	{ 95,0,1,0,0x0cfd,0x2cfd,0x00009d49,0x01af90,0x06,0x400,0x12d1 },
	{ 96,0,0,0,0x10fd,0x2cfd,0x00009d49,0x000000,0x06,0x400,0x12d1 },
	{ 97,0,1,0,0x14fd,0x2cfd,0x00003165,0x006be4,0x07,0x400,0x1980 },
	{ 98,0,0,0,0x18fd,0x2cfd,0x00003165,0x000000,0x07,0x400,0x1980 },
	{ 99,0,1,0,0x1cfd,0x2cfd,0x0000166c,0x001af9,0x08,0x400,0x1e0f },
	{ 100,0,1,0,0x00fd,0x0cfd,0x0008f051,0x0d7c80,0x09,0x400,0x035e },
	{ 101,0,1,0,0x04fd,0x0cfd,0x00023211,0x06be40,0x0a,0x400,0x0b78 },
	{ 102,0,0,0,0x08fd,0x0cfd,0x00023211,0x000000,0x0a,0x400,0x0b78 },
	{ 103,0,1,1,0x0cfd,0x4beb,0x00008281,0x01af90,0x00,0x37f,0x13e4 },
	{ 104,0,1,1,0x107c,0x4beb,0x00008281,0x000000,0x00,0x37f,0x13e4 },
	{ 105,0,0,1,0x150b,0x4cfb,0x00000209,0x008078,0x01,0x37f,0x2000 },
	{ 106,0,1,1,0x188a,0x4cfb,0x00000209,0x000000,0x01,0x37f,0x2000 },
	{ 107,0,1,1,0x1c09,0x4cfb,0x00000209,0x000000,0x01,0x37f,0x2000 },
	{ 108,0,1,1,0x1f88,0x4cfb,0x00000209,0x000000,0x01,0x37f,0x2000 },
	{ 109,0,1,1,0x0307,0x2cfb,0x00020909,0x000000,0x01,0x37f,0x0be7 },
	{ 110,0,1,1,0x0686,0x2cfb,0x00020909,0x000000,0x01,0x37f,0x0be7 },
	{ 111,0,1,1,0x0a05,0x2cfb,0x00020909,0x000000,0x01,0x37f,0x0be7 },
	{ 112,0,0,1,0x0e94,0x2e0b,0x00007f19,0x0189f0,0x02,0x37f,0x140b },
	{ 113,0,1,1,0x1213,0x2e0b,0x00007f19,0x000000,0x02,0x37f,0x140b },
	{ 114,0,0,1,0x16a2,0x2f1b,0x00001d8d,0x00618c,0x03,0x37f,0x1c76 },
	{ 115,0,1,1,0x1a21,0x2f1b,0x00001d8d,0x000000,0x03,0x37f,0x1c76 },
	{ 116,0,0,1,0x1eb0,0x302b,0x00000564,0x001829,0x04,0x37f,0x2000 },
	{ 117,0,1,1,0x022f,0x102b,0x00056457,0x000000,0x04,0x37f,0x0648 },
	{ 118,0,1,1,0x05ae,0x102b,0x00056457,0x000000,0x04,0x37f,0x0648 },
	{ 119,0,0,1,0x0a3d,0x113b,0x000220b7,0x0343a0,0x05,0x37f,0x0ba6 },
	{ 120,0,0,1,0x0ecc,0x124b,0x0000a567,0x017b50,0x06,0x37f,0x1287 },
	{ 121,0,1,1,0x124b,0x5139,0x0000a567,0x000000,0x00,0x37f,0x1287 },
	{ 122,0,0,1,0x16da,0x5249,0x0000477b,0x005dec,0x01,0x37f,0x175e },
	{ 123,0,0,1,0x1b69,0x5359,0x00001cd9,0x002aa2,0x02,0x37f,0x1c9a },
	{ 124,0,0,1,0x1ff8,0x5469,0x0000097f,0x00135a,0x03,0x37f,0x2000 },
	{ 125,0,0,1,0x0487,0x3579,0x0000b6ca,0x08c880,0x04,0x37f,0x11f3 },
	{ 126,0,1,1,0x0806,0x3579,0x0000b6ca,0x000000,0x04,0x37f,0x11f3 },
	{ 127,0,1,1,0x0b85,0x3579,0x0000b6ca,0x000000,0x04,0x37f,0x11f3 },
	{ 128,0,1,1,0x0f04,0x3579,0x0000b6ca,0x000000,0x04,0x37f,0x11f3 },
	{ 129,0,0,1,0x1393,0x3689,0x0000110a,0x00a5c0,0x05,0x37f,0x1fa4 },
	{ 130,0,1,1,0x1712,0x3689,0x0000110a,0x000000,0x05,0x37f,0x1fa4 },
	{ 131,0,1,1,0x1a91,0x3689,0x0000110a,0x000000,0x05,0x37f,0x1fa4 },
	{ 132,0,1,1,0x1e10,0x3689,0x0000110a,0x000000,0x05,0x37f,0x1fa4 },
	{ 133,0,0,1,0x029f,0x1799,0x0004d26f,0x0c3800,0x06,0x37f,0x06ed },
	{ 134,0,1,1,0x061e,0x1799,0x0004d26f,0x000000,0x06,0x37f,0x06ed },
	{ 135,0,0,1,0x0aad,0x18a9,0x0001cbcf,0x0306a0,0x07,0x37f,0x0ca0 },
	{ 136,0,0,1,0x0f3c,0x19b9,0x00006c2f,0x015fa0,0x08,0x37f,0x14fa },
	{ 137,0,1,1,0x12bb,0x19b9,0x00006c2f,0x000000,0x08,0x37f,0x14fa },
	{ 138,0,0,1,0x174a,0x1ac9,0x0000151b,0x005714,0x09,0x37f,0x1e68 },
	{ 139,0,1,1,0x1ac9,0x59b7,0x0000151b,0x000000,0x00,0x37f,0x1e68 },
	{ 140,0,1,1,0x1e48,0x59b7,0x0000151b,0x000000,0x00,0x37f,0x1e68 },
	{ 141,0,0,1,0x02d7,0x3ac7,0x00095733,0x0bc400,0x01,0x37f,0x031d },
	{ 142,0,0,1,0x0766,0x3bd7,0x00040033,0x055700,0x02,0x37f,0x0800 },
	{ 143,0,0,1,0x0bf5,0x3ce7,0x000193b3,0x026c80,0x03,0x37f,0x0d60 },
	{ 144,0,0,1,0x1084,0x3df7,0x00007a13,0x0119a0,0x04,0x37f,0x1447 },
	{ 145,0,1,1,0x1403,0x3df7,0x00007a13,0x000000,0x04,0x37f,0x1447 },
	{ 146,0,0,1,0x1892,0x3f07,0x00003453,0x0045c0,0x05,0x37f,0x192b },
	{ 147,0,0,1,0x1d21,0x4017,0x000014af,0x001fa4,0x06,0x37f,0x1e86 },
	{ 148,0,0,1,0x01b0,0x2127,0x0006525b,0x0e5d00,0x07,0x37f,0x055d },
	{ 149,0,1,1,0x052f,0x2127,0x0006525b,0x000000,0x07,0x37f,0x055d },
	{ 150,0,0,1,0x09be,0x2237,0x0002c3db,0x038e80,0x08,0x37f,0x0a23 },
	{ 151,0,0,1,0x0e4d,0x2347,0x0001268b,0x019d50,0x09,0x37f,0x0f32 },
	{ 152,0,0,1,0x12dc,0x2457,0x00006af3,0x00bb98,0x0a,0x37f,0x150b },
	{ 153,0,0,0,0x176b,0x576b,0x000015cb,0x005528,0x00,0x400,0x1e39 },
	{ 154,0,0,0,0x1b6b,0x576b,0x000015cb,0x000000,0x00,0x400,0x1e39 },
	{ 155,0,1,0,0x1f6b,0x576b,0x00000415,0x0011b6,0x01,0x400,0x2000 },
	{ 156,0,0,0,0x036b,0x376b,0x000415a3,0x000000,0x01,0x400,0x07e2 },
	{ 157,0,0,0,0x076b,0x376b,0x000415a3,0x000000,0x01,0x400,0x07e2 },
	{ 158,0,1,0,0x0b6b,0x376b,0x0001dee3,0x0236c0,0x02,0x400,0x0c64 },
	{ 159,0,1,0,0x0f6b,0x376b,0x0000c383,0x011b60,0x03,0x400,0x118f },
	{ 160,0,1,0,0x136b,0x376b,0x000035d3,0x008db0,0x04,0x400,0x1901 },
	{ 161,0,0,0,0x176b,0x376b,0x000035d3,0x000000,0x04,0x400,0x1901 },
	{ 162,0,1,0,0x1b6b,0x376b,0x00001267,0x00236c,0x05,0x400,0x1f33 },
	{ 163,0,1,0,0x1f6b,0x376b,0x000000b1,0x0011b6,0x06,0x400,0x2000 },
	{ 164,0,0,0,0x036b,0x176b,0x0000b1e8,0x000000,0x06,0x400,0x121b },
	{ 165,0,0,0,0x076b,0x176b,0x0000b1e8,0x000000,0x06,0x400,0x121b },
	{ 166,0,0,0,0x0b6b,0x176b,0x0000b1e8,0x000000,0x06,0x400,0x121b },
	{ 167,0,0,0,0x0f6b,0x176b,0x0000b1e8,0x000000,0x06,0x400,0x121b },
	{ 168,0,1,0,0x136b,0x176b,0x00002438,0x008db0,0x07,0x400,0x1b4a },
	{ 169,0,0,0,0x176b,0x576b,0x00002438,0x000000,0x00,0x400,0x1b4a },
	{ 170,0,1,0,0x1b6b,0x576b,0x000000cc,0x00236c,0x01,0x400,0x2000 },
	{ 171,0,0,0,0x1f6b,0x576b,0x000000cc,0x000000,0x01,0x400,0x2000 },
	{ 172,0,0,0,0x036b,0x376b,0x0000ccd0,0x000000,0x01,0x400,0x114b },
	{ 173,0,0,0,0x076b,0x376b,0x0000ccd0,0x000000,0x01,0x400,0x114b },
	{ 174,0,0,0,0x0b6b,0x376b,0x0000ccd0,0x000000,0x01,0x400,0x114b },
	{ 175,0,0,0,0x0f6b,0x376b,0x0000ccd0,0x000000,0x01,0x400,0x114b },
	{ 176,0,1,0,0x136b,0x376b,0x00003f20,0x008db0,0x02,0x400,0x1815 },
	{ 177,0,0,0,0x176b,0x376b,0x00003f20,0x000000,0x02,0x400,0x1815 },
	{ 178,0,1,0,0x1b6b,0x376b,0x00001bb4,0x00236c,0x03,0x400,0x1cd6 },
	{ 179,0,1,0,0x1f6b,0x376b,0x000009fe,0x0011b6,0x04,0x400,0x2000 },
	{ 180,0,1,0,0x036b,0x176b,0x00012338,0x08db00,0x05,0x400,0x0f43 },
	{ 181,0,0,0,0x076b,0x176b,0x00012338,0x000000,0x05,0x400,0x0f43 },
	{ 182,0,0,0,0x0b6b,0x176b,0x00012338,0x000000,0x05,0x400,0x0f43 },
	{ 183,0,1,0,0x0f6b,0x176b,0x000007d8,0x011b60,0x06,0x400,0x2000 },
	{ 184,0,0,0,0x136b,0x176b,0x000007d8,0x000000,0x06,0x400,0x2000 },
	{ 185,0,0,0,0x176b,0x576b,0x000007d8,0x000000,0x00,0x400,0x2000 },
	{ 186,0,0,0,0x1b6b,0x576b,0x000007d8,0x000000,0x00,0x400,0x2000 },
	{ 187,0,0,0,0x1f6b,0x576b,0x000007d8,0x000000,0x00,0x400,0x2000 },
	{ 188,0,0,0,0x036b,0x376b,0x0007d894,0x000000,0x00,0x400,0x041e },
	{ 189,0,1,0,0x076b,0x376b,0x00036b14,0x046d80,0x01,0x400,0x08e9 },
	{ 190,0,1,0,0x0b6b,0x376b,0x00013454,0x0236c0,0x02,0x400,0x0eef },
	{ 191,0,1,0,0x0f6b,0x376b,0x000018f4,0x011b60,0x03,0x400,0x1d71 },
	{ 192,0,0,0,0x136b,0x376b,0x000018f4,0x000000,0x03,0x400,0x1d71 },
	{ 193,0,0,0,0x176b,0x376b,0x000018f4,0x000000,0x03,0x400,0x1d71 },
	{ 194,0,0,0,0x1b6b,0x376b,0x000018f4,0x000000,0x03,0x400,0x1d71 },
	{ 195,0,1,0,0x1f6b,0x376b,0x0000073e,0x0011b6,0x04,0x400,0x2000 },
	{ 196,0,0,0,0x036b,0x176b,0x00073e13,0x000000,0x04,0x400,0x0494 },
	{ 197,0,1,0,0x076b,0x176b,0x0002d093,0x046d80,0x05,0x400,0x0a09 },
	{ 198,0,1,0,0x0b6b,0x176b,0x000099d3,0x0236c0,0x06,0x400,0x12f2 },
	{ 199,0,0,0,0x0f6b,0x176b,0x000099d3,0x000000,0x06,0x400,0x12f2 },
	{ 200,0,1,0,0x136b,0x176b,0x00000c23,0x008db0,0x07,0x400,0x2000 },
	{ 201,0,0,0,0x176b,0x576b,0x00000c23,0x000000,0x00,0x400,0x2000 },
	{ 202,0,0,0,0x1b6b,0x576b,0x00000c23,0x000000,0x00,0x400,0x2000 },
	{ 203,0,0,0,0x1f6b,0x576b,0x00000c23,0x000000,0x00,0x400,0x2000 },
	{ 204,0,1,0,0x036b,0x376b,0x00034815,0x08db00,0x01,0x400,0x0926 },
	{ 205,0,0,0,0x076b,0x376b,0x00034815,0x000000,0x01,0x400,0x0926 },
	{ 206,0,1,0,0x0b6b,0x376b,0x00011155,0x0236c0,0x02,0x400,0x0fa1 },
	{ 207,0,0,0,0x0f6b,0x376b,0x00011155,0x000000,0x02,0x400,0x0fa1 },
	{ 208,0,1,0,0x136b,0x376b,0x000083a5,0x008db0,0x03,0x400,0x13d8 },
	{ 209,0,1,0,0x176b,0x376b,0x00003ccd,0x0046d8,0x04,0x400,0x184d },
	{ 210,0,1,0,0x1b6b,0x376b,0x00001961,0x00236c,0x05,0x400,0x1d58 },
	{ 211,0,1,0,0x1f6b,0x376b,0x000007ab,0x0011b6,0x06,0x400,0x2000 },
	{ 212,0,0,0,0x036b,0x176b,0x0007ab45,0x000000,0x06,0x400,0x043f },
	{ 213,0,1,0,0x076b,0x176b,0x00033dc5,0x046d80,0x07,0x400,0x0938 },
	{ 214,0,1,0,0x0b6b,0x176b,0x00010705,0x0236c0,0x08,0x400,0x0fd9 },
	{ 215,0,0,0,0x0f6b,0x176b,0x00010705,0x000000,0x08,0x400,0x0fd9 },
	{ 216,0,1,0,0x136b,0x176b,0x00007955,0x008db0,0x09,0x400,0x1450 },
	{ 217,0,1,0,0x176b,0x176b,0x0000327d,0x0046d8,0x0a,0x400,0x195f },
	{ 218,0,1,1,0x1b6b,0x5a59,0x00000f11,0x00236c,0x00,0x37f,0x2000 },
	{ 219,0,1,1,0x1eea,0x5a59,0x00000f11,0x000000,0x00,0x37f,0x2000 },
	{ 220,0,0,1,0x0379,0x3b69,0x000486fd,0x0a8b00,0x01,0x37f,0x074b },
	{ 221,0,1,1,0x06f8,0x3b69,0x000486fd,0x000000,0x01,0x37f,0x074b },
	{ 222,0,0,1,0x0b87,0x3c79,0x0001ea7d,0x029c80,0x02,0x37f,0x0c40 },
	{ 223,0,0,1,0x1016,0x3d89,0x0000bb0d,0x012f70,0x03,0x37f,0x11d1 },
	{ 224,0,0,1,0x14a5,0x3e99,0x00003155,0x0089b8,0x04,0x37f,0x1982 },
	{ 225,0,1,1,0x1824,0x3e99,0x00003155,0x000000,0x04,0x37f,0x1982 },
	{ 226,0,0,1,0x1cb3,0x3fa9,0x00000f3b,0x00221a,0x05,0x37f,0x2000 },
	{ 227,0,1,1,0x0032,0x1fa9,0x000f3bfc,0x000000,0x05,0x37f,0x0049 },
	{ 228,0,0,1,0x04c1,0x20b9,0x0006ca7c,0x087180,0x06,0x37f,0x04f3 },
	{ 229,0,0,1,0x0950,0x21c9,0x0002f59c,0x03d4e0,0x07,0x37f,0x09be },
	{ 230,0,0,1,0x0ddf,0x22d9,0x0001386c,0x01bd30,0x08,0x37f,0x0edb },
	{ 231,0,0,1,0x126e,0x23e9,0x00006e54,0x00ca18,0x09,0x37f,0x14dc },
	{ 232,0,0,1,0x16fd,0x24f9,0x00001298,0x005bbc,0x0a,0x37f,0x1f23 },
	{ 233,0,1,1,0x1a7c,0x24f9,0x00001298,0x000000,0x0a,0x37f,0x1f23 },
	{ 234,0,1,1,0x1dfb,0x24f9,0x00001298,0x000000,0x0a,0x37f,0x1f23 },
	{ 235,0,0,0,0x028a,0x428a,0x00063367,0x0c6500,0x00,0x400,0x057a },
	{ 236,0,1,0,0x068a,0x428a,0x00010ba7,0x0527c0,0x01,0x400,0x0fbf },
	{ 237,0,0,0,0x0a8a,0x428a,0x00010ba7,0x000000,0x01,0x400,0x0fbf },
	{ 238,0,0,0,0x0e8a,0x428a,0x00010ba7,0x000000,0x01,0x400,0x0fbf },
	{ 239,0,1,0,0x128a,0x428a,0x000066af,0x00a4f8,0x02,0x400,0x1547 },
	{ 240,0,1,0,0x168a,0x428a,0x00001433,0x00527c,0x03,0x400,0x1ea9 },
	{ 241,0,0,0,0x1a8a,0x428a,0x00001433,0x000000,0x03,0x400,0x1ea9 },
	{ 242,0,0,0,0x1e8a,0x428a,0x00001433,0x000000,0x03,0x400,0x1ea9 },
	{ 243,0,1,0,0x028a,0x228a,0x0009e38a,0x0a4f80,0x04,0x400,0x02c8 },
	{ 244,0,1,0,0x068a,0x228a,0x0004bbca,0x0527c0,0x05,0x400,0x0709 },
	{ 245,0,1,0,0x0a8a,0x228a,0x000227ea,0x0293e0,0x06,0x400,0x0b93 },
	{ 246,0,1,0,0x0e8a,0x228a,0x0000ddfa,0x0149f0,0x07,0x400,0x10d4 },
	{ 247,0,1,0,0x128a,0x228a,0x00003902,0x00a4f8,0x08,0x400,0x18ac },
	{ 248,0,0,0,0x168a,0x228a,0x00003902,0x000000,0x08,0x400,0x18ac },
	{ 249,0,1,0,0x1a8a,0x228a,0x00000fc4,0x00293e,0x09,0x400,0x2000 },
	{ 250,0,0,0,0x1e8a,0x228a,0x00000fc4,0x000000,0x09,0x400,0x2000 },
	{ 251,0,1,0,0x028a,0x028a,0x00057498,0x0a4f80,0x0a,0x400,0x0637 },
	{ 252,0,1,1,0x068a,0x4578,0x00004cd8,0x0527c0,0x00,0x37f,0x16f3 },
	{ 253,0,1,1,0x0a09,0x4578,0x00004cd8,0x000000,0x00,0x37f,0x16f3 },
	{ 254,0,1,1,0x0d88,0x4578,0x00004cd8,0x000000,0x00,0x37f,0x16f3 },
	{ 255,0,1,1,0x1107,0x4578,0x00004cd8,0x000000,0x00,0x37f,0x16f3 },
	{ 256,0,1,1,0x1486,0x4578,0x00004cd8,0x000000,0x00,0x37f,0x16f3 },
	{ 257,0,0,1,0x1915,0x4688,0x00000d0c,0x003fcc,0x01,0x37f,0x2000 },
	{ 258,0,1,1,0x1c94,0x4688,0x00000d0c,0x000000,0x01,0x37f,0x2000 },
	{ 259,0,1,1,0x0013,0x2688,0x000d0c2a,0x000000,0x01,0x37f,0x012e },
	{ 260,0,0,1,0x04a2,0x2798,0x00046c2a,0x08a000,0x02,0x37f,0x076d },
	{ 261,0,0,1,0x0931,0x28a8,0x0000828a,0x03e9a0,0x03,0x37f,0x13e4 },
	{ 262,0,1,1,0x0cb0,0x28a8,0x0000828a,0x000000,0x03,0x37f,0x13e4 },
	{ 263,0,1,1,0x102f,0x28a8,0x0000828a,0x000000,0x03,0x37f,0x13e4 },
	{ 264,0,1,1,0x13ae,0x28a8,0x0000828a,0x000000,0x03,0x37f,0x13e4 },
	{ 265,0,0,1,0x183d,0x29b8,0x000038aa,0x0049e0,0x04,0x37f,0x18b5 },
	{ 266,0,0,1,0x1ccc,0x2ac8,0x00001724,0x002186,0x05,0x37f,0x1de0 },
	{ 267,0,0,1,0x015b,0x0bd8,0x0007ed9b,0x0f3680,0x06,0x37f,0x040e },
	{ 268,0,0,1,0x05ea,0x0ce8,0x000105db,0x06e7c0,0x07,0x37f,0x0fe0 },
	{ 269,0,1,1,0x0969,0x0ce8,0x000105db,0x000000,0x07,0x37f,0x0fe0 },
	{ 270,0,1,1,0x0ce8,0x4bd6,0x000105db,0x000000,0x00,0x37f,0x0fe0 },
	{ 271,0,0,1,0x1177,0x4ce6,0x0000170b,0x00eed0,0x01,0x37f,0x1de6 },
	{ 272,0,1,1,0x14f6,0x4ce6,0x0000170b,0x000000,0x01,0x37f,0x1de6 },
	{ 273,0,1,1,0x1875,0x4ce6,0x0000170b,0x000000,0x01,0x37f,0x1de6 },
	{ 274,0,1,1,0x1bf4,0x4ce6,0x0000170b,0x000000,0x01,0x37f,0x1de6 },
	{ 275,0,0,1,0x2083,0x4df6,0x0000056e,0x00119d,0x02,0x37f,0x2000 },
	{ 276,0,1,1,0x0402,0x2df6,0x00056ea4,0x000000,0x02,0x37f,0x063d },
	{ 277,0,0,1,0x0891,0x2f06,0x000111e4,0x045cc0,0x03,0x37f,0x0f9e },
	{ 278,0,1,1,0x0c10,0x2f06,0x000111e4,0x000000,0x03,0x37f,0x0f9e },
	{ 279,0,1,1,0x0f8f,0x2f06,0x000111e4,0x000000,0x03,0x37f,0x0f9e },
	{ 280,0,0,1,0x141e,0x3016,0x00007b04,0x0096e0,0x04,0x37f,0x143b },
	{ 281,0,0,1,0x18ad,0x3126,0x00003688,0x00447c,0x05,0x37f,0x18ed },
	{ 282,0,0,1,0x1d3c,0x3236,0x00001776,0x001f12,0x06,0x37f,0x1dcc },
	{ 283,0,0,1,0x01cb,0x1346,0x00095c18,0x0e1a00,0x07,0x37f,0x0319 },
	{ 284,0,0,1,0x065a,0x1456,0x0002f558,0x0666c0,0x08,0x37f,0x09bf },
	{ 285,0,0,1,0x0ae9,0x1566,0x00000d78,0x02e7e0,0x09,0x37f,0x2000 },
	{ 286,0,1,1,0x0e68,0x1566,0x00000d78,0x000000,0x09,0x37f,0x2000 },
	{ 287,0,1,1,0x11e7,0x1566,0x00000d78,0x000000,0x09,0x37f,0x2000 },
	{ 288,0,1,1,0x1566,0x5454,0x00000d78,0x000000,0x00,0x37f,0x2000 },
	{ 289,0,1,1,0x18e5,0x5454,0x00000d78,0x000000,0x00,0x37f,0x2000 },
	{ 290,0,1,1,0x1c64,0x5454,0x00000d78,0x000000,0x00,0x37f,0x2000 },
	{ 291,0,1,1,0x1fe3,0x5454,0x00000d78,0x000000,0x00,0x37f,0x2000 },
	{ 292,0,0,1,0x0472,0x3564,0x00049047,0x08e880,0x01,0x37f,0x073f },
	{ 293,0,0,1,0x0901,0x3674,0x00008587,0x040ac0,0x02,0x37f,0x13c3 },
	{ 294,0,1,1,0x0c80,0x3674,0x00008587,0x000000,0x02,0x37f,0x13c3 },
	{ 295,0,1,1,0x0fff,0x3674,0x00008587,0x000000,0x02,0x37f,0x13c3 },
	{ 296,0,1,1,0x137e,0x3674,0x00008587,0x000000,0x02,0x37f,0x13c3 },
	{ 297,0,0,1,0x180d,0x3784,0x00003937,0x004c50,0x03,0x37f,0x18a7 },
	{ 298,0,0,1,0x1c9c,0x3894,0x00001693,0x0022a4,0x04,0x37f,0x1e05 },
	{ 299,0,0,1,0x012b,0x19a4,0x0006dc7c,0x0fb700,0x05,0x37f,0x04e4 },
	{ 300,0,1,1,0x04aa,0x19a4,0x0006dc7c,0x000000,0x05,0x37f,0x04e4 },
	{ 301,0,0,1,0x0939,0x1ab4,0x0002f83c,0x03e440,0x06,0x37f,0x09b9 },
	{ 302,0,0,1,0x0dc8,0x1bc4,0x0001340c,0x01c430,0x07,0x37f,0x0ef0 },
	{ 303,0,0,1,0x1257,0x1cd4,0x000066cc,0x00cd40,0x08,0x37f,0x1545 },
	{ 304,0,0,1,0x16e6,0x1de4,0x000009a0,0x005d2c,0x09,0x37f,0x2000 },
	{ 305,0,1,1,0x1a65,0x1de4,0x000009a0,0x000000,0x09,0x37f,0x2000 },
	{ 306,0,1,1,0x1de4,0x5cd2,0x000009a0,0x000000,0x00,0x37f,0x2000 },
	{ 307,0,1,1,0x0163,0x3cd2,0x0009a0de,0x000000,0x00,0x37f,0x02f0 },
	{ 308,0,0,1,0x05f2,0x3de2,0x0002c2de,0x06de00,0x01,0x37f,0x0a25 },
	{ 309,0,1,1,0x0971,0x3de2,0x0002c2de,0x000000,0x01,0x37f,0x0a25 },
	{ 310,0,0,1,0x0e00,0x3ef2,0x00010f7e,0x01b360,0x02,0x37f,0x0fab },
	{ 311,0,0,1,0x128f,0x4002,0x000049de,0x00c5a0,0x03,0x37f,0x172e },
	{ 312,0,1,1,0x160e,0x4002,0x000049de,0x000000,0x03,0x37f,0x172e },
	{ 313,0,0,1,0x1a9d,0x4112,0x000018ec,0x0030f2,0x04,0x37f,0x1d72 },
	{ 314,0,0,1,0x1f2c,0x4222,0x000002b5,0x001637,0x05,0x37f,0x2000 },
	{ 315,0,1,1,0x02ab,0x2222,0x0002b5e3,0x000000,0x05,0x37f,0x0a40 },
	{ 316,0,1,1,0x062a,0x2222,0x0002b5e3,0x000000,0x05,0x37f,0x0a40 },
	{ 317,0,1,1,0x09a9,0x2222,0x0002b5e3,0x000000,0x05,0x37f,0x0a40 },
	{ 318,0,0,1,0x0e38,0x2332,0x000112b3,0x01a330,0x06,0x37f,0x0f99 },
	{ 319,0,0,1,0x12c7,0x2442,0x0000546b,0x00be48,0x07,0x37f,0x1668 },
	{ 320,0,1,1,0x1646,0x2442,0x0000546b,0x000000,0x07,0x37f,0x1668 },
	{ 321,0,0,1,0x1ad5,0x2552,0x0000254b,0x002f20,0x08,0x37f,0x1b1f },
	{ 322,0,0,1,0x1f64,0x2662,0x00000fe7,0x001564,0x09,0x37f,0x2000 },
	{ 323,0,0,1,0x03f3,0x0772,0x00063197,0x09b580,0x0a,0x37f,0x057b },
	{ 324,0,0,0,0x0882,0x4882,0x0001c997,0x046800,0x00,0x400,0x0ca7 },
	{ 325,0,0,0,0x0c82,0x4882,0x0001c997,0x000000,0x00,0x400,0x0ca7 },
	{ 326,0,1,0,0x1082,0x4882,0x0000df17,0x00ea80,0x01,0x400,0x10cc },
	{ 327,0,1,0,0x1482,0x4882,0x000069d7,0x007540,0x02,0x400,0x151a },
	{ 328,0,1,0,0x1882,0x4882,0x00002f37,0x003aa0,0x03,0x400,0x19c3 },
	{ 329,0,1,0,0x1c82,0x4882,0x000011e7,0x001d50,0x04,0x400,0x1f5b },
	{ 330,0,1,0,0x0082,0x2882,0x00033f3d,0x0ea800,0x05,0x400,0x0935 },
	{ 331,0,0,0,0x0482,0x2882,0x00033f3d,0x000000,0x05,0x400,0x0935 },
	{ 332,0,0,0,0x0882,0x2882,0x00033f3d,0x000000,0x05,0x400,0x0935 },
	{ 333,0,1,0,0x0c82,0x2882,0x00016a3d,0x01d500,0x06,0x400,0x0e01 },
	{ 334,0,1,0,0x1082,0x2882,0x00007fbd,0x00ea80,0x07,0x400,0x1404 },
	{ 335,0,1,0,0x1482,0x2882,0x00000a7d,0x007540,0x08,0x400,0x2000 },
	{ 336,0,0,0,0x1882,0x2882,0x00000a7d,0x000000,0x08,0x400,0x2000 },
	{ 337,0,0,0,0x1c82,0x2882,0x00000a7d,0x000000,0x08,0x400,0x2000 },
	{ 338,0,0,0,0x0082,0x0882,0x000a7d88,0x000000,0x08,0x400,0x0271 },
	{ 339,0,1,0,0x0482,0x0882,0x00032988,0x075400,0x09,0x400,0x095c },
	{ 340,0,0,0,0x0882,0x4882,0x00032988,0x000000,0x00,0x400,0x095c },
	{ 341,0,1,0,0x0c82,0x4882,0x00015488,0x01d500,0x01,0x400,0x0e5c },
	{ 342,0,1,0,0x1082,0x4882,0x00006a08,0x00ea80,0x02,0x400,0x1517 },
	{ 343,0,0,0,0x1482,0x4882,0x00006a08,0x000000,0x02,0x400,0x1517 },
	{ 344,0,1,0,0x1882,0x4882,0x00002f68,0x003aa0,0x03,0x400,0x19bd },
	{ 345,0,1,0,0x1c82,0x4882,0x00001218,0x001d50,0x04,0x400,0x1f4c },
	{ 346,0,1,0,0x0082,0x2882,0x000370e5,0x0ea800,0x05,0x400,0x08df },
	{ 347,0,0,0,0x0482,0x2882,0x000370e5,0x000000,0x05,0x400,0x08df },
	{ 348,0,0,0,0x0882,0x2882,0x000370e5,0x000000,0x05,0x400,0x08df },
	{ 349,0,1,0,0x0c82,0x2882,0x00019be5,0x01d500,0x06,0x400,0x0d42 },
	{ 350,0,1,0,0x1082,0x2882,0x0000b165,0x00ea80,0x07,0x400,0x121f },
	{ 351,0,1,0,0x1482,0x2882,0x00003c25,0x007540,0x08,0x400,0x185d },
	{ 352,0,1,0,0x1882,0x2882,0x00000185,0x003aa0,0x09,0x400,0x2000 },
	{ 353,0,0,0,0x1c82,0x2882,0x00000185,0x000000,0x09,0x400,0x2000 },
	{ 354,0,0,0,0x0082,0x0882,0x00018507,0x000000,0x09,0x400,0x0d97 },
	{ 355,0,0,0,0x0482,0x0882,0x00018507,0x000000,0x09,0x400,0x0d97 },
	{ 356,0,0,0,0x0882,0x4882,0x00018507,0x000000,0x00,0x400,0x0d97 },
	{ 357,0,0,0,0x0c82,0x4882,0x00018507,0x000000,0x00,0x400,0x0d97 },
	{ 358,0,1,0,0x1082,0x4882,0x00009a87,0x00ea80,0x01,0x400,0x12eb },
	{ 359,0,1,0,0x1482,0x4882,0x00002547,0x007540,0x02,0x400,0x1b20 },
	{ 360,0,0,0,0x1882,0x4882,0x00002547,0x000000,0x02,0x400,0x1b20 },
	{ 361,0,1,0,0x1c82,0x4882,0x000007f7,0x001d50,0x03,0x400,0x2000 },
	{ 362,0,0,0,0x0082,0x2882,0x0007f785,0x000000,0x03,0x400,0x0407 },
	{ 363,0,1,0,0x0482,0x2882,0x0000a385,0x075400,0x04,0x400,0x1298 },
	{ 364,0,0,0,0x0882,0x2882,0x0000a385,0x000000,0x04,0x400,0x1298 },
	{ 365,0,0,0,0x0c82,0x2882,0x0000a385,0x000000,0x04,0x400,0x1298 },
	{ 366,0,0,0,0x1082,0x2882,0x0000a385,0x000000,0x04,0x400,0x1298 },
	{ 367,0,1,0,0x1482,0x2882,0x00002e45,0x007540,0x05,0x400,0x19e1 },
	{ 368,0,0,0,0x1882,0x2882,0x00002e45,0x000000,0x05,0x400,0x19e1 },
	{ 369,0,1,0,0x1c82,0x2882,0x000010f5,0x001d50,0x06,0x400,0x1fab },
	{ 370,0,1,0,0x0082,0x0882,0x00024d93,0x0ea800,0x07,0x400,0x0b31 },
	{ 371,0,0,0,0x0482,0x0882,0x00024d93,0x000000,0x07,0x400,0x0b31 },
	{ 372,0,0,0,0x0882,0x4882,0x00024d93,0x000000,0x00,0x400,0x0b31 },
	{ 373,0,1,0,0x0c82,0x4882,0x00007893,0x01d500,0x01,0x400,0x1459 },
	{ 374,0,0,0,0x1082,0x4882,0x00007893,0x000000,0x01,0x400,0x1459 },
	{ 375,0,1,0,0x1482,0x4882,0x00000353,0x007540,0x02,0x400,0x2000 },
	{ 376,0,0,0,0x1882,0x4882,0x00000353,0x000000,0x02,0x400,0x2000 },
	{ 377,0,0,0,0x1c82,0x4882,0x00000353,0x000000,0x02,0x400,0x2000 },
	{ 378,0,0,0,0x0082,0x2882,0x00035305,0x000000,0x02,0x400,0x0912 },
	{ 379,0,0,0,0x0482,0x2882,0x00035305,0x000000,0x02,0x400,0x0912 },
	{ 380,0,0,0,0x0882,0x2882,0x00035305,0x000000,0x02,0x400,0x0912 },
	{ 381,0,1,0,0x0c82,0x2882,0x00017e05,0x01d500,0x03,0x400,0x0db2 },
	{ 382,0,1,0,0x1082,0x2882,0x00009385,0x00ea80,0x04,0x400,0x1330 },
	{ 383,0,1,0,0x1482,0x2882,0x00001e45,0x007540,0x05,0x400,0x1c53 },
	{ 384,0,0,0,0x1882,0x2882,0x00001e45,0x000000,0x05,0x400,0x1c53 },
	{ 385,0,1,0,0x1c82,0x2882,0x000000f5,0x001d50,0x06,0x400,0x2000 },
	{ 386,0,0,0,0x0082,0x0882,0x0000f5eb,0x000000,0x06,0x400,0x103c },
	{ 387,0,0,0,0x0482,0x0882,0x0000f5eb,0x000000,0x06,0x400,0x103c },
	{ 388,0,0,0,0x0882,0x4882,0x0000f5eb,0x000000,0x00,0x400,0x103c },
	{ 389,0,0,0,0x0c82,0x4882,0x0000f5eb,0x000000,0x00,0x400,0x103c },
	{ 390,0,1,0,0x1082,0x4882,0x00000b6b,0x00ea80,0x01,0x400,0x2000 },
	{ 391,0,0,0,0x1482,0x4882,0x00000b6b,0x000000,0x01,0x400,0x2000 },
	{ 392,0,0,0,0x1882,0x4882,0x00000b6b,0x000000,0x01,0x400,0x2000 },
	{ 393,0,0,0,0x1c82,0x4882,0x00000b6b,0x000000,0x01,0x400,0x2000 },
	{ 394,0,0,0,0x0082,0x2882,0x000b6bf3,0x000000,0x01,0x400,0x01f3 },
	{ 395,0,1,0,0x0482,0x2882,0x000417f3,0x075400,0x02,0x400,0x07df },
	{ 396,0,1,0,0x0882,0x2882,0x00006df3,0x03aa00,0x03,0x400,0x14e2 },
	{ 397,0,0,0,0x0c82,0x2882,0x00006df3,0x000000,0x03,0x400,0x14e2 },
	{ 398,0,0,0,0x1082,0x2882,0x00006df3,0x000000,0x03,0x400,0x14e2 },
	{ 399,0,0,0,0x1482,0x2882,0x00006df3,0x000000,0x03,0x400,0x14e2 },
	{ 400,0,1,0,0x1882,0x2882,0x00003353,0x003aa0,0x04,0x400,0x1947 },
	{ 401,0,1,0,0x1c82,0x2882,0x00001603,0x001d50,0x05,0x400,0x1e2a },
	{ 402,0,1,0,0x0082,0x0882,0x00075bfe,0x0ea800,0x06,0x400,0x047c },
	{ 403,0,1,0,0x0482,0x0882,0x000007fe,0x075400,0x07,0x400,0x2000 },
	{ 404,0,0,0,0x0882,0x4882,0x000007fe,0x000000,0x00,0x400,0x2000 },
	{ 405,0,0,0,0x0c82,0x4882,0x000007fe,0x000000,0x00,0x400,0x2000 },
	{ 406,0,0,0,0x1082,0x4882,0x000007fe,0x000000,0x00,0x400,0x2000 },
	{ 407,0,0,0,0x1482,0x4882,0x000007fe,0x000000,0x00,0x400,0x2000 },
	{ 408,0,0,0,0x1882,0x4882,0x000007fe,0x000000,0x00,0x400,0x2000 },
	{ 409,0,0,0,0x1c82,0x4882,0x000007fe,0x000000,0x00,0x400,0x2000 },
	{ 410,0,0,0,0x0082,0x2882,0x0007fe43,0x000000,0x00,0x400,0x0402 },
	{ 411,0,1,0,0x0482,0x2882,0x0000aa43,0x075400,0x01,0x400,0x125c },
	{ 412,0,0,0,0x0882,0x2882,0x0000aa43,0x000000,0x01,0x400,0x125c },
	{ 413,0,0,0,0x0c82,0x2882,0x0000aa43,0x000000,0x01,0x400,0x125c },
	{ 414,0,0,0,0x1082,0x2882,0x0000aa43,0x000000,0x01,0x400,0x125c },
	{ 415,0,1,0,0x1482,0x2882,0x00003503,0x007540,0x02,0x400,0x1917 },
	{ 416,0,0,0,0x1882,0x2882,0x00003503,0x000000,0x02,0x400,0x1917 },
	{ 417,0,1,0,0x1c82,0x2882,0x000017b3,0x001d50,0x03,0x400,0x1dbd },
	{ 418,0,1,0,0x0082,0x0882,0x00090b61,0x0ea800,0x04,0x400,0x034c },
	{ 419,0,1,0,0x0482,0x0882,0x0001b761,0x075400,0x05,0x400,0x0ce3 },
	{ 420,0,0,0,0x0882,0x4770,0x0001b761,0x000000,0x00,0x37f,0x0ce3 },
	{ 421,0,0,0,0x0c01,0x4770,0x0001b761,0x000000,0x00,0x37f,0x0ce3 },
	{ 422,0,1,0,0x1090,0x4880,0x0000a001,0x011760,0x01,0x37f,0x12b8 },
	{ 423,0,1,0,0x151f,0x4990,0x00002145,0x007ebc,0x02,0x37f,0x1bc8 },
	{ 424,0,0,0,0x189e,0x4990,0x00002145,0x000000,0x02,0x37f,0x1bc8 },
	{ 425,0,1,0,0x1d2d,0x4aa0,0x000001e2,0x001f63,0x03,0x37f,0x2000 },
	{ 426,0,0,0,0x00ac,0x2aa0,0x0001e22c,0x000000,0x03,0x37f,0x0c5a },
	{ 427,0,0,0,0x042b,0x2aa0,0x0001e22c,0x000000,0x03,0x37f,0x0c5a },
	{ 428,0,0,0,0x07aa,0x2aa0,0x0001e22c,0x000000,0x03,0x37f,0x0c5a },
	{ 429,0,0,0,0x0b29,0x2aa0,0x0001e22c,0x000000,0x03,0x37f,0x0c5a },
	{ 430,0,1,0,0x0fb8,0x2bb0,0x00009edc,0x014350,0x04,0x37f,0x12c2 },
	{ 431,0,1,0,0x1447,0x2cc0,0x00000c1c,0x0092c0,0x05,0x37f,0x2000 },
	{ 432,0,0,0,0x17c6,0x2cc0,0x00000c1c,0x000000,0x05,0x37f,0x2000 },
	{ 433,0,0,0,0x1b45,0x2cc0,0x00000c1c,0x000000,0x05,0x37f,0x2000 },
	{ 434,0,0,0,0x1ec4,0x2cc0,0x00000c1c,0x000000,0x05,0x37f,0x2000 },
	{ 435,0,1,0,0x0353,0x0dd0,0x00014ab1,0x0ad180,0x06,0x37f,0x0e87 },
	{ 436,0,0,0,0x06d2,0x0dd0,0x00014ab1,0x000000,0x06,0x37f,0x0e87 },
	{ 437,0,0,0,0x0a51,0x0dd0,0x00014ab1,0x000000,0x06,0x37f,0x0e87 },
	{ 438,0,0,0,0x0dd0,0x4cbe,0x00014ab1,0x000000,0x00,0x37f,0x0e87 },
	{ 439,0,1,0,0x125f,0x4dce,0x00007e89,0x00cc28,0x01,0x37f,0x1412 },
	{ 440,0,1,0,0x16ee,0x4ede,0x000021dd,0x005cac,0x02,0x37f,0x1bae },
	{ 441,0,0,0,0x1a6d,0x4ede,0x000021dd,0x000000,0x02,0x37f,0x1bae },
	{ 442,0,1,0,0x1efc,0x4fee,0x00000aea,0x0016f3,0x03,0x37f,0x2000 },
	{ 443,0,1,0,0x038b,0x30fe,0x00007fa3,0x0a6a80,0x04,0x37f,0x1405 },
	{ 444,0,0,0,0x070a,0x30fe,0x00007fa3,0x000000,0x04,0x37f,0x1405 },
	{ 445,0,0,0,0x0a89,0x30fe,0x00007fa3,0x000000,0x04,0x37f,0x1405 },
	{ 446,0,0,0,0x0e08,0x30fe,0x00007fa3,0x000000,0x04,0x37f,0x1405 },
	{ 447,0,0,0,0x1187,0x30fe,0x00007fa3,0x000000,0x04,0x37f,0x1405 },
	{ 448,0,1,0,0x1616,0x320e,0x00001467,0x006b3c,0x05,0x37f,0x1e9a },
	{ 449,0,0,0,0x1995,0x320e,0x00001467,0x000000,0x05,0x37f,0x1e9a },
	{ 450,0,0,0,0x1d14,0x320e,0x00001467,0x000000,0x05,0x37f,0x1e9a },
	{ 451,0,1,0,0x01a3,0x131e,0x0005eafe,0x0e7d00,0x06,0x37f,0x05bf },
	{ 452,0,0,0,0x0522,0x131e,0x0005eafe,0x000000,0x06,0x37f,0x05bf },
	{ 453,0,1,0,0x09b1,0x142e,0x0002545e,0x0396a0,0x07,0x37f,0x0b20 },
	{ 454,0,1,0,0x0e40,0x153e,0x0000b36e,0x01a0f0,0x08,0x37f,0x120e },
	{ 455,0,0,0,0x11bf,0x153e,0x0000b36e,0x000000,0x08,0x37f,0x120e },
	{ 456,0,1,0,0x164e,0x164e,0x00004c2e,0x006740,0x09,0x37f,0x1700 },
	{ 457,0,1,0,0x1add,0x1add,0x00001d50,0x002ede,0x0a,0x37f,0x1c82 },
	{ 458,0,1,1,0x1f6c,0x5f6c,0x0000080a,0x001546,0x00,0x400,0x2000 },
	{ 459,0,1,1,0x036c,0x3f6c,0x00080a62,0x000000,0x00,0x400,0x03fa },
	{ 460,0,0,1,0x076c,0x3f6c,0x00039da2,0x046cc0,0x01,0x400,0x0896 },
	{ 461,0,0,1,0x0b6c,0x3f6c,0x00016742,0x023660,0x02,0x400,0x0e0d },
	{ 462,0,0,1,0x0f6c,0x3f6c,0x00004c12,0x011b30,0x03,0x400,0x1702 },
	{ 463,0,1,1,0x136c,0x3f6c,0x00004c12,0x000000,0x03,0x400,0x1702 },
	{ 464,0,0,1,0x176c,0x3f6c,0x00000546,0x0046cc,0x04,0x400,0x2000 },
	{ 465,0,1,1,0x1b6c,0x3f6c,0x00000546,0x000000,0x04,0x400,0x2000 },
	{ 466,0,1,1,0x1f6c,0x3f6c,0x00000546,0x000000,0x04,0x400,0x2000 },
	{ 467,0,1,1,0x036c,0x1f6c,0x000546a7,0x000000,0x04,0x400,0x0668 },
	{ 468,0,0,1,0x076c,0x1f6c,0x0000d9e7,0x046cc0,0x05,0x400,0x10ef },
	{ 469,0,1,1,0x0b6c,0x1f6c,0x0000d9e7,0x000000,0x05,0x400,0x10ef },
	{ 470,0,1,1,0x0f6c,0x1f6c,0x0000d9e7,0x000000,0x05,0x400,0x10ef },
	{ 471,0,0,1,0x136c,0x1f6c,0x00004c4f,0x008d98,0x06,0x400,0x16fe },
	{ 472,0,0,1,0x176c,0x1f6c,0x00000583,0x0046cc,0x07,0x400,0x2000 },
	{ 473,0,1,1,0x1b6c,0x1f6c,0x00000583,0x000000,0x07,0x400,0x2000 },
	{ 474,0,1,1,0x1f6c,0x5f6c,0x00000583,0x000000,0x00,0x400,0x2000 },
	{ 475,0,1,1,0x036c,0x3f6c,0x000583f8,0x000000,0x00,0x400,0x0627 },
	{ 476,0,0,1,0x076c,0x3f6c,0x00011738,0x046cc0,0x01,0x400,0x0f81 },
	{ 477,0,1,1,0x0b6c,0x3f6c,0x00011738,0x000000,0x01,0x400,0x0f81 },
	{ 478,0,1,1,0x0f6c,0x3f6c,0x00011738,0x000000,0x01,0x400,0x0f81 },
	{ 479,0,0,1,0x136c,0x3f6c,0x000089a0,0x008d98,0x02,0x400,0x1396 },
	{ 480,0,0,1,0x176c,0x3f6c,0x000042d4,0x0046cc,0x03,0x400,0x17c1 },
	{ 481,0,0,1,0x1b6c,0x3f6c,0x00001f6e,0x002366,0x04,0x400,0x1c1b },
	{ 482,0,0,1,0x1f6c,0x3f6c,0x00000dbb,0x0011b3,0x05,0x400,0x2000 },
	{ 483,0,0,1,0x036c,0x1f6c,0x0004e18e,0x08d980,0x06,0x400,0x06db },
	{ 484,0,0,1,0x076c,0x1f6c,0x000074ce,0x046cc0,0x07,0x400,0x1488 },
	{ 485,0,1,1,0x0b6c,0x1f6c,0x000074ce,0x000000,0x07,0x400,0x1488 },
	{ 486,0,1,1,0x0f6c,0x1f6c,0x000074ce,0x000000,0x07,0x400,0x1488 },
	{ 487,0,1,1,0x136c,0x1f6c,0x000074ce,0x000000,0x07,0x400,0x1488 },
	{ 488,0,0,1,0x176c,0x1f6c,0x00002e02,0x0046cc,0x08,0x400,0x19e9 },
	{ 489,0,0,1,0x1b6c,0x1f6c,0x00000a9c,0x002366,0x09,0x400,0x2000 },
	{ 490,0,1,1,0x1f6c,0x5f6c,0x00000a9c,0x000000,0x00,0x400,0x2000 },
	{ 491,0,0,1,0x036c,0x3f6c,0x0001c303,0x08d980,0x01,0x400,0x0cbc },
	{ 492,0,1,1,0x076c,0x3f6c,0x0001c303,0x000000,0x01,0x400,0x0cbc },
	{ 493,0,1,1,0x0b6c,0x3f6c,0x0001c303,0x000000,0x01,0x400,0x0cbc },
	{ 494,0,0,1,0x0f6c,0x3f6c,0x0000a7d3,0x011b30,0x02,0x400,0x1271 },
	{ 495,0,0,1,0x136c,0x3f6c,0x00001a3b,0x008d98,0x03,0x400,0x1d27 },
	{ 496,0,1,1,0x176c,0x3f6c,0x00001a3b,0x000000,0x03,0x400,0x1d27 },
	{ 497,0,1,1,0x1b6c,0x3f6c,0x00001a3b,0x000000,0x03,0x400,0x1d27 },
	{ 498,0,0,1,0x1f6c,0x3f6c,0x00000888,0x0011b3,0x04,0x400,0x2000 },
	{ 499,0,1,1,0x036c,0x1f6c,0x0008881b,0x000000,0x04,0x400,0x03a2 },
	{ 500,0,0,1,0x076c,0x1f6c,0x00041b5b,0x046cc0,0x05,0x400,0x07da },
	{ 501,0,0,1,0x0b6c,0x1f6c,0x0001e4fb,0x023660,0x06,0x400,0x0c51 },
	{ 502,0,0,1,0x0f6c,0x1f6c,0x0000c9cb,0x011b30,0x07,0x400,0x1161 },
	{ 503,0,0,1,0x136c,0x1f6c,0x00003c33,0x008d98,0x08,0x400,0x185b },
	{ 504,0,1,1,0x176c,0x1f6c,0x00003c33,0x000000,0x08,0x400,0x185b },
	{ 505,0,0,1,0x1b6c,0x1f6c,0x000018cd,0x002366,0x09,0x400,0x1d7a },
	{ 506,0,0,1,0x1f6c,0x1f6c,0x0000071a,0x0011b3,0x0a,0x400,0x2000 },
	{ 507,0,1,1,0x036c,0x436c,0x00071ad1,0x000000,0x00,0x400,0x04b0 },
	{ 508,0,0,1,0x076c,0x436c,0x0002ae11,0x046cc0,0x01,0x400,0x0a51 },
	{ 509,0,0,1,0x0b6c,0x436c,0x000077b1,0x023660,0x02,0x400,0x1464 },
	{ 510,0,1,1,0x0f6c,0x436c,0x000077b1,0x000000,0x02,0x400,0x1464 },
	{ 511,0,1,1,0x136c,0x436c,0x000077b1,0x000000,0x02,0x400,0x1464 },
	{ 512,0,0,1,0x176c,0x436c,0x000030e5,0x0046cc,0x03,0x400,0x198f },
	{ 513,0,0,1,0x1b6c,0x436c,0x00000d7f,0x002366,0x04,0x400,0x2000 },
	{ 514,0,1,1,0x1f6c,0x436c,0x00000d7f,0x000000,0x04,0x400,0x2000 },
	{ 515,0,0,1,0x036c,0x236c,0x0004a5fa,0x08d980,0x05,0x400,0x0724 },
	{ 516,0,0,1,0x076c,0x236c,0x0000393a,0x046cc0,0x06,0x400,0x18a6 },
	{ 517,0,1,1,0x0b6c,0x236c,0x0000393a,0x000000,0x06,0x400,0x18a6 },
	{ 518,0,1,1,0x0f6c,0x236c,0x0000393a,0x000000,0x06,0x400,0x18a6 },
	{ 519,0,1,1,0x136c,0x236c,0x0000393a,0x000000,0x06,0x400,0x18a6 },
	{ 520,0,1,1,0x176c,0x236c,0x0000393a,0x000000,0x06,0x400,0x18a6 },
	{ 521,0,0,1,0x1b6c,0x236c,0x000015d4,0x002366,0x07,0x400,0x1e36 },
	{ 522,0,0,1,0x1f6c,0x236c,0x00000421,0x0011b3,0x08,0x400,0x2000 },
	{ 523,0,1,1,0x036c,0x436c,0x0004215c,0x000000,0x00,0x400,0x07d2 },
	{ 524,0,1,1,0x076c,0x436c,0x0004215c,0x000000,0x00,0x400,0x07d2 },
	{ 525,0,0,1,0x0b6c,0x436c,0x0001eafc,0x023660,0x01,0x400,0x0c3f },
	{ 526,0,0,1,0x0f6c,0x436c,0x0000cfcc,0x011b30,0x02,0x400,0x1135 },
	{ 527,0,0,1,0x136c,0x436c,0x00004234,0x008d98,0x03,0x400,0x17cf },
	{ 528,0,1,1,0x176c,0x436c,0x00004234,0x000000,0x03,0x400,0x17cf },
	{ 529,0,0,1,0x1b6c,0x436c,0x00001ece,0x002366,0x04,0x400,0x1c39 },
	{ 530,0,0,1,0x1f6c,0x436c,0x00000d1b,0x0011b3,0x05,0x400,0x2000 },
	{ 531,0,0,1,0x036c,0x236c,0x00044187,0x08d980,0x06,0x400,0x07a6 },
	{ 532,0,1,1,0x076c,0x236c,0x00044187,0x000000,0x06,0x400,0x07a6 },
	{ 533,0,0,1,0x0b6c,0x236c,0x00020b27,0x023660,0x07,0x400,0x0be1 },
	{ 534,0,0,1,0x0f6c,0x236c,0x0000eff7,0x011b30,0x08,0x400,0x1061 },
	{ 535,0,0,1,0x136c,0x236c,0x0000625f,0x008d98,0x09,0x400,0x1586 },
	{ 536,0,0,1,0x176c,0x236c,0x00001b93,0x0046cc,0x0a,0x400,0x1cdd },
	{ 537,0,1,1,0x1b6c,0x236c,0x00001b93,0x000000,0x0a,0x400,0x1cdd },
	{ 538,0,0,0,0x1f6c,0x5e5a,0x000009e0,0x0011b3,0x00,0x37f,0x2000 },
	{ 539,0,1,0,0x03fb,0x3f6a,0x000038e2,0x09a800,0x01,0x37f,0x18af },
	{ 540,0,0,0,0x077a,0x3f6a,0x000038e2,0x000000,0x01,0x37f,0x18af },
	{ 541,0,0,0,0x0af9,0x3f6a,0x000038e2,0x000000,0x01,0x37f,0x18af },
	{ 542,0,0,0,0x0e78,0x3f6a,0x000038e2,0x000000,0x01,0x37f,0x18af },
	{ 543,0,0,0,0x11f7,0x3f6a,0x000038e2,0x000000,0x01,0x37f,0x18af },
	{ 544,0,0,0,0x1576,0x3f6a,0x000038e2,0x000000,0x01,0x37f,0x18af },
	{ 545,0,1,0,0x1a05,0x407a,0x000002a4,0x00363e,0x02,0x37f,0x2000 },
	{ 546,0,0,0,0x1d84,0x407a,0x000002a4,0x000000,0x02,0x37f,0x2000 },
	{ 547,0,0,0,0x0103,0x207a,0x0002a453,0x000000,0x02,0x37f,0x0a66 },
	{ 548,0,0,0,0x0482,0x207a,0x0002a453,0x000000,0x02,0x37f,0x0a66 },
	{ 549,0,0,0,0x0801,0x207a,0x0002a453,0x000000,0x02,0x37f,0x0a66 },
	{ 550,0,1,0,0x0c90,0x218a,0x00007593,0x022ec0,0x03,0x37f,0x147f },
	{ 551,0,0,0,0x100f,0x218a,0x00007593,0x000000,0x03,0x37f,0x147f },
	{ 552,0,0,0,0x138e,0x218a,0x00007593,0x000000,0x03,0x37f,0x147f },
	{ 553,0,1,0,0x181d,0x229a,0x00002a17,0x004b7c,0x04,0x37f,0x1a6c },
	{ 554,0,1,0,0x1cac,0x23aa,0x000007d3,0x002244,0x05,0x37f,0x2000 },
	{ 555,0,0,0,0x002b,0x03aa,0x0007d32b,0x000000,0x05,0x37f,0x0422 },
	{ 556,0,0,0,0x03aa,0x3eab,0x0007d32b,0x000000,0x00,0x31b,0x0422 },
	{ 557,0,1,0,0x08bb,0x40a1,0x0002deeb,0x04f440,0x01,0x31b,0x09eb },
	{ 558,0,1,0,0x0dcc,0x4297,0x0000cfeb,0x020f00,0x02,0x31b,0x1134 },
	{ 559,0,0,0,0x10e7,0x4297,0x0000cfeb,0x000000,0x02,0x31b,0x1134 },
	{ 560,0,1,0,0x15f8,0x448d,0x0000500f,0x007fdc,0x03,0x31b,0x16b7 },
	{ 561,0,1,0,0x1b09,0x4683,0x00001ae9,0x003526,0x04,0x31b,0x1d01 },
	{ 562,0,1,0,0x201a,0x4879,0x000004d1,0x001618,0x05,0x31b,0x2000 },
	{ 563,0,0,0,0x0335,0x2879,0x0004d18f,0x000000,0x05,0x31b,0x06ee },
	{ 564,0,0,0,0x0650,0x2879,0x0004d18f,0x000000,0x05,0x31b,0x06ee },
	{ 565,0,1,0,0x0b61,0x2a6f,0x0001b04f,0x032140,0x06,0x31b,0x0cfb },
	{ 566,0,1,0,0x1072,0x2c65,0x0000632f,0x014d20,0x07,0x31b,0x157a },
	{ 567,0,0,0,0x138d,0x2c65,0x0000632f,0x000000,0x07,0x31b,0x157a },
	{ 568,0,1,0,0x189e,0x2e5b,0x0000125b,0x0050d4,0x08,0x31b,0x1f36 },
	{ 569,0,0,0,0x1bb9,0x2e5b,0x0000125b,0x000000,0x08,0x31b,0x1f36 },
	{ 570,0,0,0,0x1ed4,0x2e5b,0x0000125b,0x000000,0x08,0x31b,0x1f36 },
	{ 571,0,1,0,0x03e5,0x1051,0x0006e80f,0x0b7380,0x09,0x31b,0x04da },
	{ 572,0,1,0,0x08f6,0x1247,0x0002258f,0x04c280,0x0a,0x31b,0x0b99 },
	{ 573,0,1,0,0x0e07,0x4cf5,0x00002b4f,0x01fa40,0x00,0x37f,0x1a42 },
	{ 574,0,0,0,0x1186,0x4cf5,0x00002b4f,0x000000,0x00,0x37f,0x1a42 },
	{ 575,0,0,0,0x1505,0x4cf5,0x00002b4f,0x000000,0x00,0x37f,0x1a42 },
	{ 576,0,0,0,0x1884,0x4cf5,0x00002b4f,0x000000,0x00,0x37f,0x1a42 },
	{ 577,0,1,0,0x1d13,0x4e05,0x00000b5e,0x001ff1,0x01,0x37f,0x2000 },
	{ 578,0,0,0,0x0092,0x2e05,0x000b5e39,0x000000,0x01,0x37f,0x01fa },
	{ 579,0,1,0,0x0521,0x2f15,0x00037539,0x07e900,0x02,0x37f,0x08d8 },
	{ 580,0,0,0,0x08a0,0x2f15,0x00037539,0x000000,0x02,0x37f,0x08d8 },
	{ 581,0,1,0,0x0d2f,0x3025,0x00017fb9,0x01f580,0x03,0x37f,0x0dab },
	{ 582,0,1,0,0x11be,0x3135,0x00009c19,0x00e3a0,0x04,0x37f,0x12dc },
	{ 583,0,1,0,0x164d,0x3245,0x000034c5,0x006754,0x05,0x37f,0x191e },
	{ 584,0,1,0,0x1adc,0x3355,0x000005df,0x002ee6,0x06,0x37f,0x2000 },
	{ 585,0,0,0,0x1e5b,0x3355,0x000005df,0x000000,0x06,0x37f,0x2000 },
	{ 586,0,0,0,0x01da,0x1355,0x0005dfa9,0x000000,0x06,0x37f,0x05ca },
	{ 587,0,0,0,0x0559,0x1355,0x0005dfa9,0x000000,0x06,0x37f,0x05ca },
	{ 588,0,1,0,0x09e8,0x1465,0x00026aa9,0x037500,0x07,0x37f,0x0aea },
	{ 589,0,1,0,0x0e77,0x1575,0x0000d8e9,0x0191c0,0x08,0x37f,0x10f6 },
	{ 590,0,1,0,0x1306,0x1685,0x00002291,0x00b658,0x09,0x37f,0x1b90 },
	{ 591,0,0,0,0x1685,0x5573,0x00002291,0x000000,0x00,0x37f,0x1b90 },
	{ 592,0,0,0,0x1a04,0x5573,0x00002291,0x000000,0x00,0x37f,0x1b90 },
	{ 593,0,1,0,0x1e93,0x5683,0x000009ee,0x0018a3,0x01,0x37f,0x2000 },
	{ 594,0,0,0,0x0212,0x3683,0x0009ee60,0x000000,0x01,0x37f,0x02c2 },
	{ 595,0,1,0,0x06a1,0x3793,0x0003d460,0x061a00,0x02,0x37f,0x0841 },
	{ 596,0,1,0,0x0b30,0x38a3,0x00010f60,0x02c500,0x03,0x37f,0x0fab },
	{ 597,0,0,0,0x0eaf,0x38a3,0x00010f60,0x000000,0x03,0x37f,0x0fab },
	{ 598,0,1,0,0x133e,0x39b3,0x00005fd0,0x00af90,0x04,0x37f,0x15ad },
	{ 599,0,1,0,0x17cd,0x3ac3,0x00001020,0x004fb0,0x05,0x37f,0x1ff5 },
	{ 600,0,0,0,0x1b4c,0x3ac3,0x00001020,0x000000,0x05,0x37f,0x1ff5 },
	{ 601,0,0,0,0x1ecb,0x3ac3,0x00001020,0x000000,0x05,0x37f,0x1ff5 },
	{ 602,0,1,0,0x035a,0x1bd3,0x00055c15,0x0ac480,0x06,0x37f,0x0651 },
	{ 603,0,1,0,0x07e9,0x1ce3,0x000078d5,0x04e340,0x07,0x37f,0x1456 },
	{ 604,0,0,0,0x0b68,0x1ce3,0x000078d5,0x000000,0x07,0x37f,0x1456 },
	{ 605,0,0,0,0x0ee7,0x1ce3,0x000078d5,0x000000,0x07,0x37f,0x1456 },
	{ 606,0,0,0,0x1266,0x1ce3,0x000078d5,0x000000,0x07,0x37f,0x1456 },
	{ 607,0,1,0,0x16f5,0x1df3,0x00001c99,0x005c3c,0x08,0x37f,0x1ca7 },
	{ 608,0,0,0,0x1a74,0x1df3,0x00001c99,0x000000,0x08,0x37f,0x1ca7 },
	{ 609,0,1,0,0x1f03,0x1f03,0x000005c2,0x0016d7,0x09,0x37f,0x2000 },
	{ 610,0,0,0,0x0282,0x4170,0x0005c2db,0x000000,0x00,0x37f,0x05e6 },
	{ 611,0,1,0,0x0711,0x4280,0x00001adb,0x05a800,0x01,0x37f,0x1d04 },
	{ 612,0,0,0,0x0a90,0x4280,0x00001adb,0x000000,0x01,0x37f,0x1d04 },
	{ 613,0,0,0,0x0e0f,0x4280,0x00001adb,0x000000,0x01,0x37f,0x1d04 },
	{ 614,0,0,0,0x118e,0x4280,0x00001adb,0x000000,0x01,0x37f,0x1d04 },
	{ 615,0,0,0,0x150d,0x4280,0x00001adb,0x000000,0x01,0x37f,0x1d04 },
	{ 616,0,0,0,0x188c,0x4280,0x00001adb,0x000000,0x01,0x37f,0x1d04 },
	{ 617,0,0,0,0x1c0b,0x4280,0x00001adb,0x000000,0x01,0x37f,0x1d04 },
	{ 618,0,1,0,0x209a,0x4390,0x00000983,0x001158,0x02,0x37f,0x2000 },
	{ 619,0,1,0,0x0529,0x24a0,0x0001a4c7,0x07de40,0x03,0x37f,0x0d23 },
	{ 620,0,0,0,0x08a8,0x24a0,0x0001a4c7,0x000000,0x03,0x37f,0x0d23 },
	{ 621,0,0,0,0x0c27,0x24a0,0x0001a4c7,0x000000,0x03,0x37f,0x0d23 },
	{ 622,0,1,0,0x10b6,0x25b0,0x00009487,0x011040,0x04,0x37f,0x1326 },
	{ 623,0,1,0,0x1545,0x26c0,0x00001903,0x007b84,0x05,0x37f,0x1d6d },
	{ 624,0,0,0,0x18c4,0x26c0,0x00001903,0x000000,0x05,0x37f,0x1d6d },
	{ 625,0,0,0,0x1c43,0x26c0,0x00001903,0x000000,0x05,0x37f,0x1d6d },
	{ 626,0,1,0,0x20d2,0x27d0,0x00000851,0x0010b2,0x06,0x37f,0x2000 },
	{ 627,0,1,0,0x0561,0x08e0,0x0000bddd,0x079340,0x07,0x37f,0x11bb },
	{ 628,0,0,0,0x08e0,0x47ce,0x0000bddd,0x000000,0x00,0x37f,0x11bb },
	{ 629,0,0,0,0x0c5f,0x47ce,0x0000bddd,0x000000,0x00,0x37f,0x11bb },
	{ 630,0,0,0,0x0fde,0x47ce,0x0000bddd,0x000000,0x00,0x37f,0x11bb },
	{ 631,0,1,0,0x146d,0x48de,0x00002ed5,0x008f08,0x01,0x37f,0x19cf },
	{ 632,0,0,0,0x17ec,0x48de,0x00002ed5,0x000000,0x01,0x37f,0x19cf },
	{ 633,0,1,0,0x1c7b,0x49ee,0x00000b69,0x00236c,0x02,0x37f,0x2000 },
	{ 634,0,0,0,0x1ffa,0x49ee,0x00000b69,0x000000,0x02,0x37f,0x2000 },
	{ 635,0,1,0,0x0489,0x2afe,0x0002a3a6,0x08c580,0x03,0x37f,0x0a68 },
	{ 636,0,0,0,0x0808,0x2afe,0x0002a3a6,0x000000,0x03,0x37f,0x0a68 },
	{ 637,0,1,0,0x0c97,0x2c0e,0x00007786,0x022c20,0x04,0x37f,0x1466 },
	{ 638,0,0,0,0x1016,0x2c0e,0x00007786,0x000000,0x04,0x37f,0x1466 },
	{ 639,0,0,0,0x1395,0x2c0e,0x00007786,0x000000,0x04,0x37f,0x1466 },
	{ 640,0,1,0,0x1824,0x2d1e,0x00002c62,0x004b24,0x05,0x37f,0x1a1e },
	{ 641,0,1,0,0x1cb3,0x2e2e,0x00000a48,0x00221a,0x06,0x37f,0x2000 },
	{ 642,0,0,0,0x0032,0x0e2e,0x000a48ed,0x000000,0x06,0x37f,0x028e },
	{ 643,0,1,0,0x04c1,0x0f3e,0x0001d76d,0x087180,0x07,0x37f,0x0c7b },
	{ 644,0,0,0,0x0840,0x0f3e,0x0001d76d,0x000000,0x07,0x37f,0x0c7b },
	{ 645,0,0,0,0x0bbf,0x0f3e,0x0001d76d,0x000000,0x07,0x37f,0x0c7b },
	{ 646,0,1,0,0x104e,0x104e,0x0000b34d,0x012420,0x08,0x37f,0x120f },
	{ 647,0,1,0,0x14dd,0x14dd,0x00002ebd,0x008490,0x09,0x37f,0x19d2 },
	{ 648,0,0,0,0x185c,0x574a,0x00002ebd,0x000000,0x00,0x37f,0x19d2 },
	{ 649,0,1,0,0x1ceb,0x585a,0x00000de9,0x0020d4,0x01,0x37f,0x2000 },
	{ 650,0,0,0,0x006a,0x385a,0x000de932,0x000000,0x01,0x37f,0x00d0 },
	{ 651,0,1,0,0x04f9,0x396a,0x0005c832,0x082100,0x02,0x37f,0x05e1 },
	{ 652,0,1,0,0x0988,0x3a7a,0x000217d2,0x03b060,0x03,0x37f,0x0bbe },
	{ 653,0,1,0,0x0e17,0x3b8a,0x00006b22,0x01acb0,0x04,0x37f,0x1508 },
	{ 654,0,0,0,0x1196,0x3b8a,0x00006b22,0x000000,0x04,0x37f,0x1508 },
	{ 655,0,1,0,0x1625,0x3c9a,0x000000fa,0x006a28,0x05,0x37f,0x2000 },
	{ 656,0,0,0,0x19a4,0x3c9a,0x000000fa,0x000000,0x05,0x37f,0x2000 },
	{ 657,0,0,0,0x1d23,0x3c9a,0x000000fa,0x000000,0x05,0x37f,0x2000 },
	{ 658,0,0,0,0x00a2,0x1c9a,0x0000faa8,0x000000,0x05,0x37f,0x1020 },
	{ 659,0,0,0,0x0421,0x1c9a,0x0000faa8,0x000000,0x05,0x37f,0x1020 },
	{ 660,0,0,0,0x07a0,0x1c9a,0x0000faa8,0x000000,0x05,0x37f,0x1020 },
	{ 661,0,0,0,0x0b1f,0x1c9a,0x0000faa8,0x000000,0x05,0x37f,0x1020 },
	{ 662,0,0,0,0x0e9e,0x1c9a,0x0000faa8,0x000000,0x05,0x37f,0x1020 },
	{ 663,0,1,0,0x132d,0x1daa,0x00004910,0x00b198,0x06,0x37f,0x173e },
	{ 664,0,0,0,0x16ac,0x1daa,0x00004910,0x000000,0x06,0x37f,0x173e },
	{ 665,0,1,0,0x1b3b,0x1eba,0x00001d14,0x002bfc,0x07,0x37f,0x1c8e },
	{ 666,0,1,0,0x1fca,0x1fca,0x0000091d,0x0013f7,0x08,0x37f,0x2000 },
	{ 667,0,1,0,0x0459,0x0459,0x00000e2d,0x090f80,0x09,0x37f,0x2000 },
	{ 668,0,0,0,0x07d8,0x46c6,0x00000e2d,0x000000,0x00,0x37f,0x2000 },
	{ 669,0,0,0,0x0b57,0x46c6,0x00000e2d,0x000000,0x00,0x37f,0x2000 },
	{ 670,0,0,0,0x0ed6,0x46c6,0x00000e2d,0x000000,0x00,0x37f,0x2000 },
	{ 671,0,0,0,0x1255,0x46c6,0x00000e2d,0x000000,0x00,0x37f,0x2000 },
	{ 672,0,0,0,0x15d4,0x46c6,0x00000e2d,0x000000,0x00,0x37f,0x2000 },
	{ 673,0,0,0,0x1953,0x46c6,0x00000e2d,0x000000,0x00,0x37f,0x2000 },
	{ 674,0,0,0,0x1cd2,0x46c6,0x00000e2d,0x000000,0x00,0x37f,0x2000 },
	{ 675,0,0,0,0x0051,0x26c6,0x000e2d39,0x000000,0x00,0x37f,0x00b4 },
	{ 676,0,1,0,0x04e0,0x27d6,0x0005e8b9,0x084480,0x01,0x37f,0x05c1 },
	{ 677,0,1,0,0x096f,0x28e6,0x00022839,0x03c080,0x02,0x37f,0x0b92 },
	{ 678,0,1,0,0x0dfe,0x29f6,0x00007439,0x01b400,0x03,0x37f,0x148f },
	{ 679,0,0,0,0x117d,0x29f6,0x00007439,0x000000,0x03,0x37f,0x148f },
	{ 680,0,1,0,0x160c,0x2b06,0x00000841,0x006bf8,0x04,0x37f,0x2000 },
	{ 681,0,0,0,0x198b,0x2b06,0x00000841,0x000000,0x04,0x37f,0x2000 },
	{ 682,0,0,0,0x1d0a,0x2b06,0x00000841,0x000000,0x04,0x37f,0x2000 },
	{ 683,0,0,0,0x0089,0x0b06,0x0008416a,0x000000,0x04,0x37f,0x03d3 },
	{ 684,0,1,0,0x0518,0x0c16,0x00004bea,0x07f580,0x05,0x37f,0x1705 },
	{ 685,0,0,0,0x0897,0x0c16,0x00004bea,0x000000,0x05,0x37f,0x1705 },
	{ 686,0,0,0,0x0c16,0x4717,0x00004bea,0x000000,0x00,0x31b,0x1705 },
	{ 687,0,0,0,0x0f31,0x4717,0x00004bea,0x000000,0x00,0x31b,0x1705 },
	{ 688,0,0,0,0x124c,0x4717,0x00004bea,0x000000,0x00,0x31b,0x1705 },
	{ 689,0,0,0,0x1567,0x4717,0x00004bea,0x000000,0x00,0x31b,0x1705 },
	{ 690,0,1,0,0x1a78,0x490d,0x0000114a,0x003aa0,0x01,0x31b,0x1f8f },
	{ 691,0,0,0,0x1d93,0x490d,0x0000114a,0x000000,0x01,0x31b,0x1f8f },
	{ 692,0,1,0,0x02a4,0x2b03,0x0003100d,0x0e3a00,0x02,0x31b,0x098c },
	{ 693,0,0,0,0x05bf,0x2b03,0x0003100d,0x000000,0x02,0x31b,0x098c },
	{ 694,0,0,0,0x08da,0x2b03,0x0003100d,0x000000,0x02,0x31b,0x098c },
	{ 695,0,1,0,0x0deb,0x2cf9,0x00010c0d,0x020400,0x03,0x31b,0x0fbd },
	{ 696,0,1,0,0x12fc,0x2eef,0x00003595,0x00d678,0x04,0x31b,0x1908 },
	{ 697,0,0,0,0x1617,0x2eef,0x00003595,0x000000,0x04,0x31b,0x1908 },
	{ 698,0,1,0,0x1b28,0x30e5,0x00000189,0x00340c,0x05,0x31b,0x2000 },
	{ 699,0,0,0,0x1e43,0x30e5,0x00000189,0x000000,0x05,0x31b,0x2000 },
	{ 700,0,0,0,0x015e,0x10e5,0x0001892b,0x000000,0x05,0x31b,0x0d87 },
	{ 701,0,0,0,0x0479,0x10e5,0x0001892b,0x000000,0x05,0x31b,0x0d87 },
	{ 702,0,0,0,0x0794,0x10e5,0x0001892b,0x000000,0x05,0x31b,0x0d87 },
	{ 703,0,0,0,0x0aaf,0x10e5,0x0001892b,0x000000,0x05,0x31b,0x0d87 },
	{ 704,0,1,0,0x0fc0,0x12db,0x0000117b,0x0177b0,0x06,0x31b,0x1f7f },
	{ 705,0,0,0,0x12db,0x4ddc,0x0000117b,0x000000,0x00,0x31b,0x1f7f },
	{ 706,0,0,0,0x15f6,0x4ddc,0x0000117b,0x000000,0x00,0x31b,0x1f7f },
	{ 707,0,0,0,0x1911,0x4ddc,0x0000117b,0x000000,0x00,0x31b,0x1f7f },
	{ 708,0,0,0,0x1c2c,0x4ddc,0x0000117b,0x000000,0x00,0x31b,0x1f7f },
	{ 709,0,0,0,0x1f47,0x4ddc,0x0000117b,0x000000,0x00,0x31b,0x1f7f },
	{ 710,0,1,0,0x0458,0x2fd2,0x0006e38e,0x0a9800,0x01,0x31b,0x04de },
	{ 711,0,1,0,0x0969,0x31c8,0x00027c4e,0x046740,0x02,0x31b,0x0ac0 },
	{ 712,0,1,0,0x0e7a,0x33be,0x0000a7ee,0x01d460,0x03,0x31b,0x1270 },
	{ 713,0,0,0,0x1195,0x33be,0x0000a7ee,0x000000,0x03,0x31b,0x1270 },
	{ 714,0,1,0,0x16a6,0x35b4,0x00003646,0x0071a8,0x04,0x31b,0x18f4 },
	{ 715,0,1,0,0x1bb7,0x37aa,0x00000706,0x002f40,0x05,0x31b,0x2000 },
	{ 716,0,0,0,0x1ed2,0x37aa,0x00000706,0x000000,0x05,0x31b,0x2000 },
	{ 717,0,0,0,0x01ed,0x17aa,0x0007063c,0x000000,0x05,0x31b,0x04c1 },
	{ 718,0,1,0,0x06fe,0x19a0,0x000054bc,0x06b180,0x06,0x31b,0x1663 },
	{ 719,0,0,0,0x0a19,0x19a0,0x000054bc,0x000000,0x06,0x31b,0x1663 },
	{ 720,0,0,0,0x0d34,0x19a0,0x000054bc,0x000000,0x06,0x31b,0x1663 },
	{ 721,0,0,0,0x104f,0x19a0,0x000054bc,0x000000,0x06,0x31b,0x1663 },
	{ 722,0,0,0,0x136a,0x19a0,0x000054bc,0x000000,0x06,0x31b,0x1663 },
	{ 723,0,1,0,0x187b,0x1b96,0x000001f8,0x0052c4,0x07,0x31b,0x2000 },
	{ 724,0,0,0,0x1b96,0x5697,0x000001f8,0x000000,0x00,0x31b,0x2000 },
	{ 725,0,0,0,0x1eb1,0x5697,0x000001f8,0x000000,0x00,0x31b,0x2000 },
	{ 726,0,0,0,0x01cc,0x3697,0x0001f80f,0x000000,0x00,0x31b,0x0c18 },
	{ 727,0,0,0,0x04e7,0x3697,0x0001f80f,0x000000,0x00,0x31b,0x0c18 },
	{ 728,0,0,0,0x0802,0x3697,0x0001f80f,0x000000,0x00,0x31b,0x0c18 },
	{ 729,0,0,0,0x0b1d,0x3697,0x0001f80f,0x000000,0x00,0x31b,0x0c18 },
	{ 730,0,1,0,0x102e,0x388d,0x00009b4f,0x015cc0,0x01,0x31b,0x12e4 },
	{ 731,0,1,0,0x153f,0x3a83,0x00000a57,0x0090f8,0x02,0x31b,0x2000 },
	{ 732,0,0,0,0x185a,0x3a83,0x00000a57,0x000000,0x02,0x31b,0x2000 },
	{ 733,0,0,0,0x1b75,0x3a83,0x00000a57,0x000000,0x02,0x31b,0x2000 },
	{ 734,0,0,0,0x1e90,0x3a83,0x00000a57,0x000000,0x02,0x31b,0x2000 },
	{ 735,0,0,0,0x01ab,0x1a83,0x000a57e4,0x000000,0x02,0x31b,0x0286 },
	{ 736,0,1,0,0x06bc,0x1c79,0x00035824,0x06ffc0,0x03,0x31b,0x0909 },
	{ 737,0,1,0,0x0bcd,0x1e6f,0x00006f44,0x02e8e0,0x04,0x31b,0x14d0 },
	{ 738,0,0,0,0x0ee8,0x1e6f,0x00006f44,0x000000,0x04,0x31b,0x14d0 },
	{ 739,0,0,0,0x1203,0x1e6f,0x00006f44,0x000000,0x04,0x31b,0x14d0 },
	{ 740,0,1,0,0x1714,0x2065,0x000005c0,0x006984,0x05,0x31b,0x2000 },
	{ 741,0,0,0,0x1a2f,0x2065,0x000005c0,0x000000,0x05,0x31b,0x2000 },
	{ 742,0,0,0,0x1d4a,0x2065,0x000005c0,0x000000,0x05,0x31b,0x2000 },
	{ 743,0,0,0,0x0065,0x3a4f,0x0005c090,0x000000,0x00,0x2c2,0x05e9 },
};
uint8_t *streams[]=
{
	stream0,
	stream1,
	stream2,
	stream3,
	stream4,
	stream5,
	stream6,
	stream7,
};

struct KnownState *knownstates[]=
{
	knownstates0,
	knownstates1,
	knownstates2,
	knownstates3,
	knownstates4,
	knownstates5,
	knownstates6,
	knownstates7,
};

int numsteps[]=
{
	256,
	333,
	68,
	294,
	293,
	274,
	296,
	743,
};



